-- ---------------------------------------------------------------
--	Spread Sompare
-- ---------------------------------------------------------------
--	$Header
-- ---------------------------------------------------------------
--
--	Program Name	:	SpreadCompare
--	Function		:	script executed by Geneos netprobe
--					:	to compare the BID/ASK Prices across two feeds
--	Author			:	Richard Gould
--	Language		:	Lua (5.1)
--	Creation		:	01/02/2015
--	History			:
--
--	01/02/2015	0.00 -> 1.00	RG	Creation
--	25/06/2015	1.00 -> 2.00	RG	Adaption
--
-- ---------------------------------------------------------------
local Program =	 "SpreadCompare"
local Version =	 "2"
local Revision = "01"
-- ---------------------------------------------------------------

local sa = require 'geneos.sampler'
local md = require 'geneos.marketdata'
local sc = require 'geneos.sampler.commands'
local he = require 'geneos.helpers'

-- ---------------------------------------------------------------
-- Config file is pathed by parameter from Gateway

local ConfigDir = sa.params[ 'ConfigDir' ]

package.path = package.path .. ";" .. ConfigDir .. "?.lua"

local config = require "config"

-- ---------------------------------------------------------------
-- These are the default values

local baseFeedName = sa.params[ 'baseFeedName' ] or "bloomberg"
local altFeedName  = sa.params[ 'altFeedName' ]	 or "bloomberg"
local viewName	   = sa.params[ 'viewName' ]	 or "Spread Compare"

if ( altFeedName == baseFeedName ) then
	altFeedName = baseFeedName .. "A"
end

-- Timezone offset
config.offset = config.offset or 0

-- End of Defaults
-- ---------------------------------------------------------------

-- define columns and publish view

local cols = {	"instrument", "avgbaseBid", "avgbaseAsk", "baseTicks", "baseSpread", "avgaltBid", "avgaltAsk", "altTicks", "altSpread", "BidCompare", "AskCompare" }
local view = assert( sa.createView( viewName, cols ) )

-- ---------------------------------------------------------------
-- Load the feed parameters and start the feeds

local baseFeed = md.addFeed( viewName, config[ baseFeedName ] )
baseFeed:start()

local altFeed = md.addFeed( viewName, config[ altFeedName ] )
altFeed:start()

-- ---------------------------------------------------------------
local instruments = config[ baseFeedName ].instruments

local totalTicks = 0

local SampleTime = he.formatTime( "%d/%m/%y %H:%M:%S", he.gettimeofday() )

view.headline.totalTicks = totalTicks
view.headline.baseFeed	 = baseFeedName
view.headline.altFeed	 = altFeedName
view.headline.Now		 = SampleTime
view.headline.Version	 = Version .. "." .. Revision
view:publish()
-- ---------------------------------------------------------------

-- utility function to round down values
local function roundDown( value, dp )
	dpformat = "%4." .. dp .. "f"
	local mult = 10^( dp or 0 )
	if ( value == 0 ) then
		return string.format( dpformat, 0 )
	end
	return not value or string.format( dpformat, math.ceil( value * mult - 0.5 ) / mult )
end

local function getFeedStats( feed, instrument_name )
	local sumBid, sumAsk, sumSpread, avgBid, avgAsk, avgSpread = 0, 0, 0, 0, 0, 0
	local TickCount = 0

	local values = {}

	local tick = feed:getTicks( instrument_name )

	if tick then
		local spread = 0
		-- ---------------------------------------------------------------
		-- Create Function within function as ticks are local
		-- ---------------------------------------------------------------
		local getStatsFromTick = function( TicksTable )
			local count = 0
			while TicksTable.next do
				spread	   = TicksTable.field.Ask - TicksTable.field.Bid
				sumSpread  = sumSpread + spread
				sumAsk	   = sumAsk + TicksTable.field.Ask
				sumBid	   = sumBid + TicksTable.field.Bid

				count = count + 1
				TicksTable = TicksTable.next
			end
			return {
				avgBid		 = ( sumBid	   / count ),
				avgAsk		 = ( sumAsk	   / count ),
				avgSpread	 = ( sumSpread / count ),
				Ticks		 = count
			}
		end
		-- ---------------------------------------------------------------

		statsTable = getStatsFromTick( tick )

		while tick.next do
			TickCount = TickCount + 1
			tick = tick.next
		end

		values = {
			avgBid     = statsTable.avgBid,
			avgAsk     = statsTable.avgAsk,
			avgSpread  = statsTable.avgSpread,
			Ticks      = statsTable.Ticks
		}

	end
	return values, TickCount	-- also return the number of ticks observed
end

local function calculateRowStats( instrument_name )
	local rowValues = {}
	local c1, c2

	local baseStats, c1 = getFeedStats( baseFeed, instrument_name )
	local altStats,  c2 = getFeedStats( altFeed,  instrument_name )

	local TickCount = c1 + c2

	local BidCompare, AskCompare = 0, 0

	if not baseStats.avgBid or not altStats.avgBid then
		BidCompare = 0
	else
		BidCompare = baseStats.avgBid - altStats.avgBid
	end
	if not baseStats.avgAsk or not altStats.avgAsk then
		AskCompare = 0
	else
		AskCompare = baseStats.avgAsk - altStats.avgAsk
	end

	rowValues = {
		avgbaseBid = roundDown( baseStats.avgBid, 4 ),
		avgbaseAsk = roundDown( baseStats.avgAsk, 4 ),
		baseTicks  = baseStats.Ticks,
		avgaltBid  = roundDown( altStats.avgBid, 4 ),
		avgaltAsk  = roundDown( altStats.avgAsk, 4 ),
		altTicks   = altStats.Ticks,
		baseSpread = roundDown( baseStats.avgSpread, 4 ),
		altSpread  = roundDown( altStats.avgSpread, 4 ),
		BidCompare = BidCompare,
		AskCompare = AskCompare
		}
	return rowValues, TickCount
end

sa.doSample = function()

	SampleTime = he.formatTime( "%d/%m/%y %H:%M:%S", he.gettimeofday() )

	for name, i in pairs( instruments ) do
		local rowResult, TickCount = calculateRowStats( name )
		view.row[ name ] = rowResult
		totalTicks = totalTicks + TickCount
	end
	view.headline.totalTicks = totalTicks					-- update headline with latest tally
	view.headline.Now = SampleTime

	assert( view:publish() )						-- publish the new view content
end

-- create the command and add to the headline
local Reset = sc.newDefinition()
	:addHeadlineTarget( view, "totalTicks" )		-- added to headline that matches the name 'totalTicks'

local resetTotalTicks = function( target, args )
	totalTicks = 0									-- reset counter
	view.headline.totalTicks = totalTicks			-- update the view
	assert(view:publish())							-- publish out updated view
end
sa.publishCommand( "Reset Tick Count", resetTotalTicks,	Reset )
