-- ###################################################################################
-- #                             dataview_publishing.lua                             #
-- ###################################################################################
-- # This example script demonstrates different ways of publishing Geneos dataviews. #
-- # Many of the tips shown are in fact syntactic sugar for Lua language constructs. #
-- ###################################################################################

local gs = require "geneos.sampler"



-- Create a view
local cols = {"rowName", "mul10", "mod10"}
local view = assert(gs.createView("MyView", cols))

-- Another view sharing the same columns
local sameColsView = assert(gs.createView("SameColumns", cols))

-- Another view with (different) columns declared inline
local subrowsView = assert(gs.createView("subrows", {"row","col1", "col2", "col3"}))



-- Creating, updating and removing headlines
view.headline["status"] = "connected" -- Just assign to a headline to create it
view.headline["total"] = 0
view.headline["total"] = 13           -- To update a headline, just assign a new value
view.headline["total"] = nil          -- Assigning the value nil will remove the headline.
view.headline = {}                    -- Remove all headlines by assigning the empty table.

-- Note: The samplingStatus headline will always be present on a dataview
--       and cannot be removed, although it's value can be set to empty.



-- Lua syntactic sugar for table operations
-- If the table key strings are valid Lua identifiers, they can be used directly (not as strings).
view.headline.total = 17.4
-- This also applies to table initialisation
-- Long form:
view.headline = { ["total"]=22.8, ["status"]="disconnected" }
-- Short form:
view.headline = { total=22.8, status="disconnected" }



-- Creating, updating and removing table rows
-- Working with rows works as for headlines, except that rows must be a table of values.

-- Rows can be either an array:
-- (with values defined in the same order as column names)
view.row.vals = { "value 1", "value 2" }
-- Or a mapping:
-- (from column name to cell value)
view.row.static = { mul10="count * 10", mod10="c % 10" }

view.row.vals = nil                  -- remove a row
view.row.static.mod10 = "count % 10" -- replace a specific column value



-- Sub-rows are displayed 'indented' in Geneos UI
-- This convention is typically used to display "summary" and "detail" data in the same view.
-- The top-level row shows the summary or aggregate data.
subrowsView.row["top"]      = { 8, 7, 4 }
subrowsView.row["top#sub1"] = { 2, 7, 3 }
subrowsView.row["top#sub2"] = { 6, 0, 1 }

subrowsView.row["head"]      = { 1, 0, 0 }
subrowsView.row["head#sub1"] = { 1, 0, 0 }

assert(subrowsView:publish())



-- Publish initial view content above
view.headline.samplingStatus = "OK"
assert(view:publish())



-- Count the number of samples
-- and each sample update the 'calc' row with new calculated values
local count = 0
gs.doSample = function()
	count = count + 1
	view.headline.count = count
	view.row["calc"] = { count * 10, count % 10 }
	assert(view:publish())
end
