--- #####################
--- feed_subscription.lua
--  #####################
--  Example code to demonstrate how to subscribe to a market
--  data feed and process tick data from that feed

-- Import packages
local gs = require "geneos.sampler"
local md = require "geneos.marketdata"

-- Setup feed configuration
local feedConfiguration = {}                        -- Feed configuration is a Lua table

feedConfiguration.feed = {                          -- Generic feed configuration
	type = "example",
	["library.filename"] = "flm-feed-example.so",   -- verbose syntax due to '.' character in lib file
	verbose = "false",                              -- log verbosity
}

feedConfiguration.example = {                       -- feed specific configuration
	publishingPeriod = 1000                         -- Defines period between published tick updates
}

feedConfiguration.instruments = {                   -- define the set of instruments to subscribe to
	GOOG = "DATA_SERVICE.GOOG",                     -- map of names to instrument codes
	IBM  = "DATA_SERVICE.IBM"
}

feedConfiguration.fields = {                        -- define the set of fields to subscribe to
	Trade = "TRADE_PRICE",                          -- map of names to field codes
	Bid   = "BID",
	Ask   = "ASK" ,
	Time  = "QUOTIM"
}

-- Create and start the feed
local exampleFeed = md.addFeed("Example-Feed", feedConfiguration)  -- Create a local feed using config from above
exampleFeed:start()                                                -- Start the feed

-- Function to unsubscribe from one instrument and subscribe to another
local function replaceSubscription(removeInst, addInst)
	if removeInst then
		exampleFeed:unsubscribe(removeInst)
	end
	if addInst then
		exampleFeed:subscribe(addInst, "DATA_SERVICE."..addInst)
	end
end

-- Utility function to print tick content
local function printTick(tick)
	print("-----------------------")      -- Each tick is a table
	print("Got tick for: " .. tick.inst)
	print("Trade " .. tick.field.Trade)   -- Subscribed fields appear
	print("Ask   " .. tick.field.Ask)     -- in a nested table called 'field'
	print("Time  " .. tick.field.Time)
	print("Bid   " .. tick.field.Bid)
end

-- Print this sample's ticks for given instrument
local function printTickStream(instName)
	local ticks = exampleFeed:getTicks(instName)
	while ticks do         
		printTick(ticks)
		ticks = ticks.next          -- ticks are returned as a linked list
	end
end

-- doSample() is called periodically by the Netprobe
local count = 0
local otherInst = "IBM"
gs.doSample = function()
	printTickStream("GOOG")
	printTickStream(otherInst)

	count = count + 1
	if count == 3 then
		replaceSubscription("IBM", "AAPL")
		otherInst = "AAPL"
	end
end
