--- ------------------------------------------------------------------------------
--- latency_calculation.lua
--
--  Example code to demonstrate how to call the latency algorithm
--  and retrieve tick statistics
--  
--  This work is licensed under a Creative Commons Attribution 3.0 Unported Licence
--- --------------------------------------------------------------------------------
-- Import packages
local gs = require "geneos.sampler"
local lt = require "geneos.latency"
local lf = require "geneos.latency.files"

-- Define fields and instruments
local fields = { Trade="TRDPRC_1", Change="NETCHNG_1", Bid="BID", BidSize="BIDSIZE", Ask="ASK", AskSize="ASKSIZE" }
local insts  = { Bund="IDN_SELECTFEED.Bund", DAX="IDN_SELECTFEED.DAX", Eurostoxx="IDN_SELECTFEED.Eurostoxx" }

-- Using an RFA base data source
local rfaBase =
{	
	feed = { type = "rfa", ["library.filename"] = "flm-feed-rfa.so"},
	rfa  = { configFile = "RFA.cfg", session = "OMMFeedBase", connectionType = "OMM" },
	instruments = insts,
	fields = fields
}

-- Using an RFA alt data source
local rfaAlt =
{	
	feed = { type = "rfa", ["library.filename"] = "flm-feed-rfa.so"},
	rfa  = { configFile = "RFA.cfg", session = "OMMFeedAlt", connectionType = "OMM" },
	instruments = insts,
	fields = fields
}

-- Create and configure the latency context
local ctx = lt.newContext() 			-- Create the context object here
	:setBase("Base-Eurex", rfaBase)		-- Register the base feed
	:addFeed("Alt-Eurex",  rfaAlt)		-- Add an alternate feed

-- Create tick loggers and start the context
local historyLogger = lf.newTickHistoryFile(gs.name..".%y%m%d.ticks")
local matchedTickLogger 
	= lf.newMatchedTicksFile(ctx:getConfiguredNames(), gs.name..".%y%m%d.csv")

ctx:addWriter(historyLogger:getWriter())	-- Register the history logger's tick writer method
	:addWriter(matchedTickLogger:getWriter())	-- and the same for the matched tick logger
	:start()								-- Start the subscriptions

-- Create a view to display the data
local view = gs.createView("LATENCY", {"feed", "status", "numTicks", "numMatches", "minLatency", "maxLatency" })
view.headline.baselineFeed = "Base-Eurex"
view:publish()							-- Publish a stats view, similar to FLM Latency view

local function formatMillisec(value)    -- Format nil value as blank string
    return value and string.format("%0.1f", value*1000) or ""
end

local count = 0
gs.doSample = function()
	count = count + 1
	print("[latency ] sample " .. count)

	ctx:sample()								-- Process the sample
	local mBase = ctx:getMetrics("Base-Eurex")	-- Grab metrics from the base feed
	local mAlt  = ctx:getMetrics("Alt-Eurex")	-- Grab metrics from the altenate feed

	view.row = {}
	view.row["Base-Eurex"] = {
		ctx:getFeedStatus("Base-Eurex"),		-- Status of the base feed
		mBase.numTicks,							-- Add remaining stats to row
		mBase.matches,
		"",										-- The base feed has no latency stats
		""
	}

	view.row["Alt-Eurex"]  = {
		ctx:getFeedStatus("Alt-Eurex"),
		mAlt.numTicks, 							-- Add remaining stats to row
		mAlt.matches, 
		formatMillisec(mAlt.latMin), 
		formatMillisec(mAlt.latMax), 
	}
	
	view:publish()								-- publish the updated view to Geneos
end
