((document) => {
    // Initialize the script execution
    let init = () => {
      // Check if the document is already loaded
      if (document.readyState !== 'loading') {
        eventHandler(); // If loaded, run the event handler
      } else {
        // If not loaded yet, wait for the DOMContentLoaded event to trigger the event handler
        document.addEventListener('DOMContentLoaded', eventHandler);
      }
    },
  
    // Event handler to process elements with the 'highlight' class
    eventHandler = () => {
      // Iterate over all elements with the 'highlight' class
      document.querySelectorAll('.highlight').forEach((highlightDiv) => {
          // Create the copy button for each code block
          createCopyButton(highlightDiv);
          
          // Add the click event listener to the copy button after it has been created and inserted
          const button = highlightDiv.querySelector('.copy-code-button');
          button.addEventListener('click', () => copyCodeToClipboard(button, highlightDiv));
      });
    },
  
    // Create and insert a copy button into the given highlightDiv
    createCopyButton = (highlightDiv) => {
      let button = document.createElement('button');
      button.className = 'copy-code-button'; // Assign a class to the button
      button.type = 'button'; // Set the button type
      button.setAttribute('title', 'Copy code'); // add tool tip
      button.appendChild(createSvg()); // Add the icon to the button
  
      // Insert the button at the beginning of the highlightDiv if it doesn't already exist
      if (!highlightDiv.querySelector('.copy-code-button')) {
        highlightDiv.insertBefore(button, highlightDiv.firstChild);
      }
    },
  
    // Copy the code content to the clipboard and update the button state
    copyCodeToClipboard = (button, highlightDiv) => {
      // Select the code block inside the highlightDiv
      const codeToCopy = highlightDiv.querySelector('.chroma > code').innerText;
      try {
        // Attempt to use the Clipboard API to write the code to the clipboard
        if (navigator.clipboard) {
          navigator.clipboard.writeText(codeToCopy);
        }
      } catch (_) {
        // If the Clipboard API is not supported, log an error message
        console.log('Browser not supported');
      } finally {
        // Change the button text to indicate success
        button.textContent = 'Copied!';
        // After 2 seconds, reset the button text to its original state
        setTimeout(() => {
          button.innerText = ''; // Clear the text
          button.appendChild(createSvg()); // Reinsert the original icon
        }, 2000);
      }
    },
  
    // Create the SVG icon for the copy button
    createSvg = () => {
      let icon = document.createElementNS('http://www.w3.org/2000/svg', 'svg'),
          useElem = document.createElementNS('http://www.w3.org/2000/svg', 'use');
  
      // Add classes and set the icon reference
      icon.classList.add('icon', 'content_copy');
      useElem.setAttributeNS('http://www.w3.org/1999/xlink', 'xlink:href', '#content_copy'); // Set the icon's reference, can be found in core-docs/core/themes/itrs/assets/sprites/material.svg
      icon.appendChild(useElem); // Append the icon to the SVG
  
      return icon; // Return the created icon
    };
  
    // Initialize the script
    init();
  
    // Expose the copy code functions for external usage
    exposeCopyCode = { 
      copyCode: init, // Allow external scripts to call the init function
      copyCodeToClipboard: copyCodeToClipboard // Expose copy function for testing or usage outside
    }
  
  })(document);
  