const LHNTopics = []; //array container for all topic element in LHN

const parentAncestors = (link) => { // function to get parent names
    try {
        let currentElement = link?.closest('.nav-menu')?.previousElementSibling;
        let parentElements = [];
        let parentName = [];

        // Traverse up the DOM to collect all parent elements and their labels
        while (currentElement && currentElement.tagName.toLowerCase() === 'label') {
            const prevElement = currentElement.closest('.nav-menu')?.previousElementSibling;
            if (!currentElement.parentNode) break; // Ensure parent node exists
            parentElements.push(currentElement.parentNode);

            // innerHTML is used because innerText and textContent does not work properly on iOS devices
            const label = currentElement.parentNode.getElementsByTagName('label')[0];
            if (label) {
                parentName.push(label.innerHTML);
            }

            if (!prevElement) break; // Stop if there's no previous sibling
            currentElement = prevElement;
        }

        return [parentElements, parentName];
    } catch (error) {
        console.error('Error in parentAncestors:', error);
        return [[], []]; // Return empty arrays in case of error
    }
};

try {
    document.querySelectorAll('.nav-link.ref-nav').forEach(link => { // collect all topic element in LHN, referencing starts at the actual topic element
        try {
            // Store topic data including title, URL, and parent hierarchy
            LHNTopics.push({
                title: link.textContent.trim(),
                url: link.href,
                parentAncestors: parentAncestors(link),
            });
        } catch (error) {
            console.error('Error processing link:', link, error);
        }
    });
} catch (error) {
    console.error('Error querying nav links:', error);
}

// Fuse JS Setup, for reference: https://www.fusejs.io/api/options.html
const fuseOption = {
    includeScore: true, // Include relevance score in results
    shouldSort: true, // Sort results by relevance
    keys: [{
            name: 'title',
            weight: 150, // Prioritize title matches
        },
        {
            name: 'parentAncestors',
            weight: 50
        }
    ],
    threshold: 0.3, // Lower threshold means stricter matching
    distance: 400 // Maximum distance to consider for fuzzy matching, threshold(0.3) * distance(400) = 120 characters max distance
};

// Initialize Fuse.js with LHN topics and options
let LHNFuse;
try {
    LHNFuse = new Fuse(LHNTopics, fuseOption);
} catch (error) {
    console.error('Error initializing Fuse.js:', error);
}

// Debounce function to delay execution of search logic
const debounce = (callback, wait) => {
    let timeoutId = null;
    return (...args) => {
        window.clearTimeout(timeoutId); // Clear previous timeout
        timeoutId = window.setTimeout(() => {
            callback.apply(null, args);
        }, wait);
    };
}

// Manual matching for inputs that fuse js fails to match, consider adding more if necessary
const specialCases = { 
    'rest api': 'API (REST)',
    'restapi': 'API (REST)',
};

const inputFilter = document.querySelector('#inputFilter');
const filterIcon = document.querySelector('.sidebar-filter-icon');
const clearBtn = document.querySelector('.sidebar-clear-button');
const noResultMsg = document.querySelector('#no-result');

const filterVisibility = (element, isVisible) => {
    try {
        isVisible ? element?.classList.remove("element-invisible") : element?.classList.add("element-invisible");
    } catch (error) {
        console.error('Error updating filter visibility:', error);
    }
};

// Triggered on input change: debounce search logic and update UI
if (inputFilter) {
    inputFilter.addEventListener('input', (e) => {
        try {
            showFilterLoader();
            hideFilterResult();

            filterVisibility(filterIcon, false) // Hide filter icon during typing
            filterVisibility(clearBtn, true)  // Show clear button
            handleInput(e); // Pass input to Fuse.js search
        } catch (error) {
            console.error('Error handling input event:', error);
        }
    });
}

// Debounced function to handle input processing
const handleInput = debounce((e) => { 
    try {
        const filterInputTerm = e.target.value.trim().toLowerCase();

        // Handle predefined special cases
        if (specialCases[filterInputTerm]) { 
            const specialResult = LHNTopics.find(
                topic => topic.title === specialCases[filterInputTerm]
            );
            if (specialResult) {
                updateVisibility([{
                    item: specialResult
                }], filterInputTerm); // Wrap in array for uniformity
                return;
            };
        };
        
        // If no special case, proceed with Fuse.js
        const results = LHNFuse.search(filterInputTerm); 
        updateVisibility(results, filterInputTerm); // pass results
    } catch (error) {
        console.error('Error in handleInput:', error);
    }
}, 500); // 500ms debounce interval

// function to clear input
const clearFilter = () => {
    try {
        inputFilter.value = '';
        hideFilterLoader();
        hideFilterResult();

        filterVisibility(filterIcon, true)
        filterVisibility(clearBtn, false)
    } catch (error) {
        console.error('Error clearing filter:', error);
    }
}

if (clearBtn) {
    clearBtn.addEventListener('click', clearFilter);
}

// Update visibility based on filter search results
function updateVisibility(results, filterInputTerm) {
    try {
        hideFilterLoader();
        showFilterResult();

        const globalSearchInput = document.getElementById('global-search-input');
        if (globalSearchInput) {
            globalSearchInput.innerText = filterInputTerm; //display filter text input in global search item
        }

        const resultsContainer = document.querySelectorAll('.filter-result');
        
        resultsContainer.forEach(item => item.remove()); // Clear previous created element results
        results.forEach((result) => { // append and display the created elements from results to the result container
            const resultElement = createResultElement(result);
            if (resultElement) {
                document.querySelector('.filter-results-container').appendChild(resultElement);
            }
        });
        
        results.length === 0 ? filterVisibility(noResultMsg, true) : filterVisibility(noResultMsg, false)

        if (filterInputTerm === '') { // clear the filter when input term is empty
            clearFilter();
        };
    } catch (error) {
        console.error('Error updating visibility:', error);
    }
}

const createResultElement = (result) => { // function to create elements based on the fuse.js results
    try {
        const resultItem = document.createElement('li'); // create the list placeholder for topic results
        resultItem.classList.add('filter-result');

        const resultTitle = document.createElement('a'); // create element title with corresponding URL
        resultTitle.classList.add('filter-result-title');
        resultTitle.href = result.item.url; 
        resultTitle.innerText = result.item.title;

        const resultParentTitle = document.createElement('a'); // create element title for parent labels
        resultParentTitle.classList.add('filter-result-parent-title');
        resultParentTitle.href = result.item.url;

        result.item.parentAncestors[0].forEach((parent, index, array) => { // iterate through parentAncestors to display the parent topics labels
            const parentLabel = document.createElement('span');
            parentLabel.innerText =
                parent.getElementsByTagName('label')[0].innerHTML + (index === array.length - 1 ? '' : ' > '); // add arrows
            resultParentTitle.appendChild(parentLabel);
        });

        resultItem.appendChild(resultTitle); // append the title to list
        resultItem.appendChild(resultParentTitle); // append the parent labels to list
        return resultItem;
    } catch (error) {
        console.error('Error creating result element:', error);
        return null;
    }
};

// Function to handle 'Esc' key press to hide results
const handleEscapeKey = (e) => {
    try {
        if (e.key === 'Escape') {
            hideFilterResult();
        }
    } catch (error) {
        console.error('Error handling escape key:', error);
    }
}

// UI Show/Hide Functions for Filter Results and Loader

const filterResultsPlaceholder = document.querySelector('.filter-results-placeholder');
const filterResultsLoader = document.querySelector('.filter-results-placeholder-loading');

const showFilterResult = () => {
    try {
        filterVisibility(filterResultsPlaceholder, true)
        filterResultsPlaceholder.style.maxHeight = `${computeHeight()}px`;
        inputFilter.addEventListener('keydown', handleEscapeKey); // listen for 'esc' key when filter result is active
    } catch (error) {
        console.error('Error showing filter result:', error);
    }
}

const hideFilterResult = () => {
    try {
        filterVisibility(filterResultsPlaceholder, false)

        inputFilter.removeEventListener('keydown', handleEscapeKey); // remove listener for 'esc' key when filter result is not active to prevent unnecessary events
    } catch (error) {
        console.error('Error hiding filter result:', error);
    }
}

const showFilterLoader = () => {
    try {
        filterVisibility(filterResultsLoader, true)
    } catch (error) {
        console.error('Error showing filter loader:', error);
    }
}

const hideFilterLoader = () => {
    try {
        filterVisibility(filterResultsLoader, false)
    } catch (error) {
        console.error('Error hiding filter loader:', error);
    }
}

// Event listeners for interactions outside the filter results

let isInsideFilterResult = false; //checks if click is outside filter result

// Track whether the pointer is inside the filter results
if (filterResultsPlaceholder) {
    filterResultsPlaceholder.addEventListener('mouseenter', () => {
        isInsideFilterResult = true;
    });

    filterResultsPlaceholder.addEventListener('mouseleave', () => {
        isInsideFilterResult = false;
    });
}

// Hide filter results when input loses focus and pointer is outside
if (inputFilter) {
    inputFilter.addEventListener('focusout', () => {
        if (!isInsideFilterResult) {
            hideFilterResult();
        }
    });

    // Show filter results when input is interacted with
    inputFilter.addEventListener('mousedown', (e) => {
        if (e.target.value !== '') {
            showFilterResult();
        }
    });
}

// Smooth scrolling to the active topic in the LHN
try {
    const activeElement = document.querySelector('.nav-link.ref-nav.is-active'); // get the active selected topic element in LHN
        activeElement?.scrollIntoView({ // scroll the LHN to the selected topic
            behavior: 'smooth',
            block: 'nearest'
        });
    
} catch (error) {
    console.error('Error scrolling to active element:', error);
}

const computeHeight = () =>{
    try {
        let navHeight = document.querySelector(".nav-scrollable")?.offsetHeight || 0;
        let externalLinksHeight = document.querySelector(".nav-external-links")?.offsetHeight || 0;

        return navHeight + externalLinksHeight + 40;
    } catch (error) {
        console.error('Error computing height:', error);
        return 0;
    }
}

// trigger search with product
const searchTrigger = document.getElementsByClassName("search-trigger element-invisible");
const globalTrigger = document.querySelector(".global-search-btn");

if (globalTrigger && searchTrigger.length > 0) {
    globalTrigger.addEventListener("click", ()=>{
        searchTrigger[0].click();
    });
}