/**
 * Content Cleaning Module
 * Handles cleaning and transforming content for PDF export, including media replacement
 */

(function (document) {
    'use strict';

    // Styling constants for media replacement elements
    const MEDIA_LINK_BUTTON_STYLES = {
        padding: '10px',
        backgroundColor: '#070032',
        borderRadius: '6px',
        color: 'white'
    };

    const MEDIA_CONTAINER_CENTER_STYLES = {
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
        width: '100%'
    };

    const MEDIA_CONTAINER_WITH_MARGIN_STYLES = {
        ...MEDIA_CONTAINER_CENTER_STYLES,
        margin: '20px 0'
    };

    /**
     * Cleans and transforms content for PDF export
     * @param {Node} content - The content node to clean
     * @returns {Node} - The cleaned content clone
     */
    const cleanContent = (content) => {
        const contentClone = content.cloneNode(true);
        
        // Remove all anchor-toc-visible classes from the clone for consistent layout
        const clonedAnchors = contentClone.querySelectorAll('.anchor-bookmark');
        clonedAnchors.forEach((anchor) => {
            anchor.classList.remove('anchor-toc-visible');
        });
        
        const storylaneFrames = contentClone.querySelectorAll(".sl-frame");
        const expandTableBtn = contentClone.querySelectorAll(".table-modal-btn");
        const downloadPdfBtn = contentClone.querySelector(".pdf-download-btn");
        const nextprevBtn = contentClone.querySelector(".nav-prev-next");
        const videos = contentClone.querySelectorAll(".video");
        const gifs = contentClone.querySelectorAll("img[src*='.gif']");

        //open all accordion
        const accordions = contentClone.querySelectorAll(".accordian-block");
        accordions.forEach((accordion) => {
            accordion.open = true;
        });

        //prevents text overlapping in tables - matches table-layout.js logic
        const table = contentClone.getElementsByTagName("table");
        table.forEach((t)=>{
            t.style.tableLayout = "fixed";
            t.style.wordBreak = "break-word";
            
            // Fix link wrapping in tables - links need explicit word breaking
            const tableLinks = t.querySelectorAll('a');
            tableLinks.forEach((link) => {
                link.style.wordBreak = "break-all"; // Break at any character for URLs

            });

            //fix overlapping text in table data cell
            const tableData = t.querySelectorAll('td');
            tableData.forEach((d)=>{
                d.style.wordWrap = "break-word";
            })
        });
        

        // Remove all storylane elements
        const doc = contentClone.ownerDocument || document;
        storylaneFrames.forEach((storylane) => {
            if (storylane.parentNode) {
                
                const storylaneURL = doc.createElement('a');

                const removeElement = storylane.parentNode.querySelector('.sl-embed');
                
                // Check if removeElement exists and has children before accessing
                if (!removeElement || !removeElement.children || removeElement.children.length === 0) {
                    return;
                }

                Object.assign(storylane.style, MEDIA_CONTAINER_WITH_MARGIN_STYLES);

                storylaneURL.href = removeElement.children[0]?.src;
                storylaneURL.textContent = 'Click to view Storylane demo';
                Object.assign(storylaneURL.style, MEDIA_LINK_BUTTON_STYLES);

                storylane.appendChild(storylaneURL);
                removeElement.remove();

            }
        });

        videos.forEach((video)=>{
            // Check if video has children before accessing
            if (!video.children || video.children.length === 0) {
                return;
            }

            if (!video.parentNode) {
                return;
            }

            const videoURL = doc.createElement('a');
            Object.assign(video.parentNode.style, MEDIA_CONTAINER_CENTER_STYLES);

            // Check if first child exists and has a valid src
            if (!video.children[0] || !video.children[0].src) {
                return;
            }

            videoURL.href = video.children[0].src;
            videoURL.textContent = 'Click here to watch video';
            Object.assign(videoURL.style, MEDIA_LINK_BUTTON_STYLES);

            video.parentNode.replaceChild(videoURL, video);

        })

        gifs.forEach((gif)=>{

            if (!gif.parentNode) {
                return;
            }

            const gifContainer = doc.createElement('div');
            const gifURL = doc.createElement('a');
            
            if(!gifURL.href || !gif.src){
                return;
            }


            gifURL.href = gif.src;
            gifURL.textContent = 'Click to view GIF';
            Object.assign(gifURL.style, MEDIA_LINK_BUTTON_STYLES, { fontSize: '1rem' });

            Object.assign(gifContainer.style, MEDIA_CONTAINER_WITH_MARGIN_STYLES);


            gifContainer.appendChild(gifURL);
            gif.parentNode.replaceChild(gifContainer, gif);

        })
        
        if (downloadPdfBtn) {
        contentClone.removeChild(downloadPdfBtn);
        }
        if (nextprevBtn) {
        contentClone.removeChild(nextprevBtn);
        }
        
        // Remove scroll-to-top button (only appears when user scrolls, so check for null)
        const scrollMeBtn = contentClone.querySelector('#scrollMe.scrollMe');
        if (scrollMeBtn && scrollMeBtn.parentNode) {
            scrollMeBtn.parentNode.removeChild(scrollMeBtn);
        }
        
        expandTableBtn.forEach((e) => {
            if (e.parentNode) {
                e.parentNode.removeChild(e);
            }
        });
        
        // Convert SVG elements using <use> references to inline SVGs for PDF export
        // This must be done after other cleaning operations
        if (window.PDFSvgProcessing && window.PDFSvgProcessing.convertSvgUseToInline) {
            window.PDFSvgProcessing.convertSvgUseToInline(contentClone);
        }
        
        // Remove any standalone SVG sprite containers (not needed in PDF)
        const svgSprites = contentClone.querySelectorAll('.svg-sprite, svg.svg-sprite');
        svgSprites.forEach((sprite) => {
            if (sprite.parentNode) {
                sprite.parentNode.removeChild(sprite);
            }
        });

        return contentClone;
    };

    // Export to global scope
    window.PDFContentCleaning = {
        cleanContent
    };

})(document);

