/**
 * PDF Download Main Module
 * Orchestrates PDF generation by coordinating all sub-modules
 * 
 * Dependencies (must be loaded before this file):
 * - pdf-svg-processing.js
 * - pdf-content-cleaning.js
 * - pdf-image-processing.js
 * - pdf-styling.js
 * - pdf-header-footer.js
 * - pdf-generation.js
 */

(function (document) {
    'use strict';

    /**
     * Checks if content contains tables with 5 or more columns
     * @param {Node} content - The content element to check
     * @returns {boolean} - True if wide tables are found
     */
    const hasWideTables = (content) => {
        const tables = content.querySelectorAll('table');
        for (const table of tables) {
            // Check the first row (header or first data row) to determine column count
            const firstRow = table.querySelector('tr');
            if (firstRow) {
                const columns = firstRow.querySelectorAll('th, td');
                if (columns.length >= 5) {
                    return true;
                }
            }
        }
        return false;
    };

    /**
     * Shows a warning modal for wide tables
     * @param {Function} onConfirm - Callback when user clicks "Download anyway"
     * @param {Function} onCancel - Callback when user clicks "Cancel"
     */
    const showWideTableWarning = (onConfirm, onCancel) => {
        // Create modal overlay
        const overlay = document.createElement('div');
        overlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10001;
        `;

        // Create modal content
        const modal = document.createElement('div');
        modal.style.cssText = `
            background: white;
            border-radius: 0.625rem;
            padding: 30px;
            max-width: 350px;
            width: 90%;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        `;

        // Create warning icon/text
        const warningText = document.createElement('div');
        warningText.style.cssText = `
            font-size: 18px;
            font-weight: bold;
            color: #333;
            margin-bottom: 15px;
            justify-self: center;
        `;
        warningText.textContent = 'Wide tables detected';

        // Create message
        const message = document.createElement('div');
        message.style.cssText = `
            font-size: 14px;
            color: #070032;
            margin-bottom: 25px;
            line-height: 1.5;
        `;
        message.textContent = 'This topic contains tables with 5 or more columns. Wide tables may not render well in PDF format and could cause layout issues.';

        // Create button container
        const buttonContainer = document.createElement('div');
        buttonContainer.style.cssText = `
            display: flex;
            gap: 10px;
            justify-content: space-evenly;
        `;

        // Create Cancel button
        const cancelBtn = document.createElement('button');
        cancelBtn.textContent = 'Cancel';
        cancelBtn.style.cssText = `
            padding: 10px 20px;
            background-color: #ffff;
            border: 1px solid #070032;
            border-radius: 0.5rem;
            cursor: pointer;
            font-size: 14px;
            color: #070032;
        `;
        cancelBtn.addEventListener('click', () => {
            document.body.removeChild(overlay);
            if (onCancel) onCancel();
        });
        cancelBtn.addEventListener('mouseenter', () => {
            cancelBtn.style.backgroundColor = '#e8e8e8';
        });
        cancelBtn.addEventListener('mouseleave', () => {
            cancelBtn.style.backgroundColor = '#ffff';
        });

        // Create Download anyway button
        const downloadBtn = document.createElement('button');
        downloadBtn.textContent = 'Download anyway';
        downloadBtn.style.cssText = `
            padding: 10px 20px;
            background-color: #070032;
            border: none;
            border-radius: 0.5rem;
            cursor: pointer;
            font-size: 14px;
            color: white;
            font-weight: bold;
        `;
        downloadBtn.addEventListener('click', () => {
            document.body.removeChild(overlay);
            if (onConfirm) onConfirm();
        });
        downloadBtn.addEventListener('mouseenter', () => {
            downloadBtn.style.backgroundColor = '#0a0040';
        });
        downloadBtn.addEventListener('mouseleave', () => {
            downloadBtn.style.backgroundColor = '#070032';
        });

        // Assemble modal
        buttonContainer.appendChild(cancelBtn);
        buttonContainer.appendChild(downloadBtn);
        modal.appendChild(warningText);
        modal.appendChild(message);
        modal.appendChild(buttonContainer);
        overlay.appendChild(modal);

        // Add to DOM
        document.body.appendChild(overlay);

        // Close on overlay click (outside modal)
        overlay.addEventListener('click', (e) => {
            if (e.target === overlay) {
                document.body.removeChild(overlay);
                if (onCancel) onCancel();
            }
        });
    };

    /**
     * Processes content and generates PDF
     * @param {Node} content - The content element to process
     */
    const processAndGeneratePDF = (content) => {
        try {
            // Clean content using the content cleaning module
            if (!window.PDFContentCleaning || !window.PDFContentCleaning.cleanContent) {
                console.error('[PDF] PDFContentCleaning module not loaded');
                alert('PDF functionality is not available. Please refresh the page.');
                return;
            }

            const contentClone = window.PDFContentCleaning.cleanContent(content);

            // Apply styling
            if (window.PDFStyling && window.PDFStyling.addSimplePdfStyling) {
                window.PDFStyling.addSimplePdfStyling(contentClone);
            }

            // Preprocess images (try to render in PDF, fallback to links if CORS prevents loading)
            // Continue with PDF generation even if image preprocessing fails
            if (window.PDFImageProcessing && window.PDFImageProcessing.preprocessImagesForPDF) {
                window.PDFImageProcessing.preprocessImagesForPDF(contentClone)
                    .catch((error) => {
                        console.error('[PDF] Error during image preprocessing:', error);
                        console.error('[PDF] Error details:', {
                            message: error?.message || 'Unknown error',
                            stack: error?.stack || 'No stack trace',
                            name: error?.name || 'Unknown error type'
                        });
                    })
                    .finally(() => {
                        // Download PDF using the generation module
                        if (window.PDFGeneration && window.PDFGeneration.downloadPDF) {
                            window.PDFGeneration.downloadPDF(contentClone);
                        } else {
                            console.error('[PDF] PDFGeneration module not loaded');
                            alert('PDF generation is not available. Please refresh the page.');
                        }
                    });
            } else {
                // If image processing module is not available, proceed directly to PDF generation
                if (window.PDFGeneration && window.PDFGeneration.downloadPDF) {
                    window.PDFGeneration.downloadPDF(contentClone);
                } else {
                    console.error('[PDF] PDFGeneration module not loaded');
                    alert('PDF generation is not available. Please refresh the page.');
                }
            }
        } catch (error) {
            console.error('[PDF] Error during content processing', error);
            alert('An error occurred while processing the content for PDF. Please check the console for details.');
        }
    };

    // Initialize the script when DOM is ready
    let init = function () {
        if (document.readyState !== 'loading') {
            eventHandler(); // DOM already loaded
        } else {
            document.addEventListener('DOMContentLoaded', eventHandler); // Wait for DOM to load
        }
    };

    const eventHandler = function () {
        const downloadBtn = document.querySelector(".pdf-download-btn");
        const content = document.querySelector(".content");

        if (!downloadBtn || !content) {
            return;
        }

        downloadBtn.addEventListener("click", () => {
            // Check for wide tables before processing
            if (hasWideTables(content)) {
                showWideTableWarning(
                    () => {
                        // User confirmed - proceed with PDF generation
                        processAndGeneratePDF(content);
                    },
                    () => {
                        // User cancelled - do nothing
                    }
                );
            } else {
                // No wide tables - proceed directly
                processAndGeneratePDF(content);
            }
        });
    };

    // Call the init function to start the process
    init();

})(document);
