/**
 * PDF Generation Module
 * Handles PDF generation, chunking, and related utilities
 */

(function (document) {
    'use strict';

    /**
     * Estimates the content height for chunking decisions
     * @param {Node} contentClone - The content clone to measure
     * @returns {Promise<number>} - The estimated content height in pixels
     */
    const estimateContentHeight = async (contentClone) => {
        const tempContainer = document.createElement('div');
        tempContainer.style.cssText = `
            position: absolute;
            visibility: hidden;
            width: 816px;
            left: -9999px;
            top: 0;
        `;
        document.body.appendChild(tempContainer);
        
        const measureClone = contentClone.cloneNode(true);
        tempContainer.appendChild(measureClone);
        void measureClone.offsetHeight;
        await new Promise(resolve => setTimeout(resolve, 100));
        
        const totalContentHeight = measureClone.scrollHeight;
        document.body.removeChild(tempContainer);
        
        return totalContentHeight;
    };

    /**
     * Determines if chunking should be used based on content height
     * @param {number} contentHeight - The content height in pixels
     * @returns {boolean} - True if chunking should be used
     */
    const shouldUseChunking = (contentHeight) => {
        const maxChunkHeight = 18000; // pixels (15 pages * 1200px)
        return contentHeight > maxChunkHeight;
    };

    /**
     * Splits content into chunks for large PDFs
     * @param {Node} contentClone - The content clone to split
     * @param {number} maxChunkHeight - Maximum height per chunk in pixels
     * @returns {Array<Node>} - Array of content chunks
     */
    const splitContentIntoChunks = (contentClone, maxChunkHeight = 18000) => {
        const chunks = [];
        const children = Array.from(contentClone.children);
        const breakPointSelectors = ['h1', 'h2', 'h3', 'section', 'article', '.page-break-before'];
        
        const originalClasses = contentClone.className;
        const originalStyles = contentClone.getAttribute('style') || '';
        const originalAttributes = {};
        Array.from(contentClone.attributes).forEach(attr => {
            if (attr.name !== 'style') {
                originalAttributes[attr.name] = attr.value;
            }
        });
        
        const tempContainer = document.createElement('div');
        tempContainer.style.cssText = `
            position: absolute;
            visibility: hidden;
            width: 816px;
            left: -9999px;
        `;
        if (originalStyles) {
            tempContainer.setAttribute('style', tempContainer.style.cssText + '; ' + originalStyles);
        }
        document.body.appendChild(tempContainer);
        
        let currentChunk = contentClone.cloneNode(false);
        Object.keys(originalAttributes).forEach(attrName => {
            currentChunk.setAttribute(attrName, originalAttributes[attrName]);
        });
        if (originalStyles) {
            currentChunk.setAttribute('style', originalStyles);
        }
        currentChunk.className = originalClasses;
        
        let currentChunkHeight = 0;
        
        for (let i = 0; i < children.length; i++) {
            const child = children[i];
            const measureClone = child.cloneNode(true);
            tempContainer.appendChild(measureClone);
            void measureClone.offsetHeight;
            const elementHeight = measureClone.scrollHeight || 100;
            tempContainer.removeChild(measureClone);
            
            const wouldExceed = currentChunkHeight + elementHeight > maxChunkHeight;
            const isBreakPoint = breakPointSelectors.some(selector => {
                return child.matches && child.matches(selector);
            });
            
            if (wouldExceed && currentChunk.children.length > 0) {
                if (isBreakPoint) {
                    chunks.push(currentChunk);
                    currentChunk = contentClone.cloneNode(false);
                    Object.keys(originalAttributes).forEach(attrName => {
                        currentChunk.setAttribute(attrName, originalAttributes[attrName]);
                    });
                    if (originalStyles) {
                        currentChunk.setAttribute('style', originalStyles);
                    }
                    currentChunk.className = originalClasses;
                    currentChunk.appendChild(child.cloneNode(true));
                    currentChunkHeight = elementHeight;
                } else {
                    currentChunk.appendChild(child.cloneNode(true));
                    currentChunkHeight += elementHeight;
                    chunks.push(currentChunk);
                    currentChunk = contentClone.cloneNode(false);
                    Object.keys(originalAttributes).forEach(attrName => {
                        currentChunk.setAttribute(attrName, originalAttributes[attrName]);
                    });
                    if (originalStyles) {
                        currentChunk.setAttribute('style', originalStyles);
                    }
                    currentChunk.className = originalClasses;
                    currentChunkHeight = 0;
                }
            } else {
                currentChunk.appendChild(child.cloneNode(true));
                currentChunkHeight += elementHeight;
            }
        }
        
        if (currentChunk.children.length > 0) {
            chunks.push(currentChunk);
        }
        
        document.body.removeChild(tempContainer);
        return chunks;
    };

    /**
     * Updates the loading progress indicator
     * @param {HTMLElement} loadingText - The loading text element
     * @param {number} current - Current chunk number
     * @param {number} total - Total number of chunks
     */
    const updateLoadingProgress = (loadingText, current, total) => {
        if (loadingText) {
            loadingText.textContent = `Generating PDF... (${current} of ${total} sections)`;
        }
    };

    /**
     * Downloads PDF from content clone
     * @param {Node} contentClone - The cleaned content clone
     */
    const downloadPDF = (contentClone) => {
        if (typeof html2pdf === 'undefined') {
            alert('PDF download is not available. Please refresh the page and try again.');
            return;
        }

        let jsPDF = null;
        if (typeof window.jsPDF !== 'undefined') {
            jsPDF = window.jsPDF;
        } else if (typeof window.jspdf !== 'undefined') {
            if (window.jspdf.jsPDF) {
                jsPDF = window.jspdf.jsPDF;
            } else {
                jsPDF = window.jspdf;
            }
        }

        if (!jsPDF) {
            alert('PDF download is not available. Please refresh the page and try again.');
            return;
        }

        const filename = window.PDFHeaderFooter ? window.PDFHeaderFooter.generateFilename() : 'documentation';
        
        const opt = {
            margin: [0.75, 0.5, 0.75, 0.5],
            filename: `${filename}.pdf`,
            image: {
                type: 'jpeg',
                quality: 1
            },
            html2canvas: {
                scale: 2,
                useCORS: true,
                allowTaint: false,
                logging: true,
                foreignObjectRendering: false,
                ignoreElements: (element) => {
                    if (element.tagName === 'IMG') {
                        const src = element.getAttribute('src') || element.src;
                        if (src && src.startsWith('data:')) {
                            return false;
                        }
                        if (element.hasAttribute('crossOrigin') || element.crossOrigin) {
                            return false;
                        }
                    }
                    return false;
                },
                onclone: (clonedDoc) => {
                    const images = clonedDoc.querySelectorAll('img');
                    images.forEach((img) => {
                        const src = img.getAttribute('src') || img.src;
                        if (src && !src.startsWith('data:')) {
                            if (!img.hasAttribute('crossOrigin') && !img.crossOrigin) {
                                console.warn('[PDF] html2canvas: Image without crossOrigin attribute may fail CORS:', src);
                            }
                        }
                    });
                }
            },
            jsPDF: {
                unit: 'in',
                format: 'legal',
                orientation: 'portrait'
            },
            pagebreak: {
                mode: ['css', 'legacy'],
                before: '.page-break-before',
                after: '.page-break-after',
                avoid: ['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'table', 'td', 'th', 'tr', 'tbody', 'thead', 'pre', '.hint', '.chroma', 'p', 'li', 'ul', 'ol']
            },
            enableLinks: true
        };

        try {
            // Store original overflow values to restore later
            const originalOverflow = document.body.style.overflow;
            const originalHtmlOverflow = document.documentElement.style.overflow;
            
            const loadingOverlay = document.createElement('div');
            loadingOverlay.id = 'pdf-loading-overlay';
            loadingOverlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0, 0, 0, 0.7);
                display: flex;
                justify-content: center;
                align-items: center;
                z-index: 10000;
                flex-direction: column;
            `;
            
            const loadingSpinner = document.createElement('div');
            loadingSpinner.style.cssText = `
                border: 4px solid #f3f3f3;
                border-top: 4px solid #070032;
                border-radius: 50%;
                width: 50px;
                height: 50px;
                animation: spin 1s linear infinite;
                margin-bottom: 20px;
            `;
            
            const loadingText = document.createElement('div');
            loadingText.textContent = 'Generating PDF...';
            loadingText.style.cssText = `
                color: white;
                font-size: 18px;
                font-family: Arial, sans-serif;
            `;
            
            const style = document.createElement('style');
            style.textContent = `
                @keyframes spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
            `;
            document.head.appendChild(style);
            
            loadingOverlay.appendChild(loadingSpinner);
            loadingOverlay.appendChild(loadingText);
            document.body.appendChild(loadingOverlay);
            
            // Scroll to top and disable scrolling during PDF generation
            window.scrollTo(0, 0);
            document.body.style.overflow = 'hidden';
            document.documentElement.style.overflow = 'hidden';
            
            setTimeout(async () => {
                try {
                    const topicTitle = window.PDFHeaderFooter ? window.PDFHeaderFooter.getTopicTitle() : 'Documentation';
                    const logoData = window.PDFHeaderFooter ? await window.PDFHeaderFooter.loadLogoAsBase64() : null;
                    
                    const contentHeight = await estimateContentHeight(contentClone);
                    const useChunking = shouldUseChunking(contentHeight);
                    
                    let finalPdf;
                    
                    if (useChunking) {
                        const chunks = splitContentIntoChunks(contentClone);
                        const headerFooterOpt = { ...opt };
                        headerFooterOpt.margin = [1.4, 0.5, 1.2, 0.5];
                        
                        let worker = html2pdf().set(headerFooterOpt);
                        
                        for (let i = 0; i < chunks.length; i++) {
                            updateLoadingProgress(loadingText, i + 1, chunks.length);
                            const chunkIndex = i;
                            const isLastChunk = chunkIndex === chunks.length - 1;
                            
                            worker = worker.from(chunks[chunkIndex]).toContainer().toCanvas().toPdf().get('pdf').then((pdf) => {
                                if (!isLastChunk) {
                                    pdf.addPage();
                                }
                            });
                        }
                        
                        finalPdf = await worker.get('pdf');
                        
                        if (!finalPdf || !finalPdf.internal) {
                            throw new Error('Failed to generate merged PDF from chunks');
                        }
                    } else {
                        const headerFooterOpt = { ...opt };
                        headerFooterOpt.margin = [1.4, 0.5, 1.2, 0.5];
                        
                        const worker = html2pdf().set(headerFooterOpt).from(contentClone);
                        finalPdf = await worker.toPdf().get('pdf');
                        
                        if (!finalPdf || !finalPdf.internal) {
                            throw new Error('Invalid jsPDF instance from html2pdf');
                        }
                    }
                    
                    const totalPages = finalPdf.internal.getNumberOfPages();
                    
                    if (totalPages === 0) {
                        throw new Error('PDF has no pages');
                    }
                    
                    if (window.PDFHeaderFooter && window.PDFHeaderFooter.addHeaderAndFooterInReservedSpace) {
                        window.PDFHeaderFooter.addHeaderAndFooterInReservedSpace(finalPdf, topicTitle, logoData);
                    }
                    
                    const pdfBlob = finalPdf.output('blob');
                    const url = URL.createObjectURL(pdfBlob);
                    const link = document.createElement('a');
                    link.href = url;
                    link.download = `${filename}.pdf`;
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                    URL.revokeObjectURL(url);
                    
                    if (loadingOverlay.parentNode) {
                        loadingOverlay.parentNode.removeChild(loadingOverlay);
                    }
                    // Re-enable scrolling
                    document.body.style.overflow = originalOverflow;
                    document.documentElement.style.overflow = originalHtmlOverflow;
                } catch (error) {
                    console.error('[PDF] Error during PDF generation:', error);
                    console.error('[PDF] Error details:', {
                        message: error?.message || 'Unknown error',
                        stack: error?.stack || 'No stack trace',
                        name: error?.name || 'Unknown error type',
                        cause: error?.cause || null
                    });
                    
                    const errorMessage = error?.message?.toLowerCase() || '';
                    const errorStack = error?.stack?.toLowerCase() || '';
                    if (errorMessage.includes('cors') || errorMessage.includes('cross-origin') || 
                        errorStack.includes('cors') || errorStack.includes('cross-origin')) {
                        console.error('[PDF] CORS error detected during PDF generation');
                    }
                    
                    try {
                        console.warn('[PDF] Attempting fallback PDF generation without headers/footers');
                        const fallbackWorker = html2pdf().set(opt).from(contentClone);
                        await fallbackWorker.save();
                        if (loadingOverlay.parentNode) {
                            loadingOverlay.parentNode.removeChild(loadingOverlay);
                        }
                        // Re-enable scrolling
                        document.body.style.overflow = originalOverflow;
                        document.documentElement.style.overflow = originalHtmlOverflow;
                        alert('PDF generated but headers/footers could not be added.');
                    } catch (error2) {
                        console.error('[PDF] Fallback PDF generation also failed:', error2);
                        if (loadingOverlay.parentNode) {
                            loadingOverlay.parentNode.removeChild(loadingOverlay);
                        }
                        // Re-enable scrolling
                        document.body.style.overflow = originalOverflow;
                        document.documentElement.style.overflow = originalHtmlOverflow;
                        alert('Failed to generate PDF. Refresh the page and try again.');
                    }
                }
            }, 100);
        } catch (error) {
            const loadingOverlay = document.getElementById('pdf-loading-overlay');
            if (loadingOverlay && loadingOverlay.parentNode) {
                loadingOverlay.parentNode.removeChild(loadingOverlay);
            }
            // Re-enable scrolling (use try-catch in case variables aren't defined)
            try {
                document.body.style.overflow = originalOverflow || '';
                document.documentElement.style.overflow = originalHtmlOverflow || '';
            } catch (e) {
                document.body.style.overflow = '';
                document.documentElement.style.overflow = '';
            }
            alert('Failed to generate PDF. Please check the console for details.');
        }
    };

    // Export to global scope
    window.PDFGeneration = {
        downloadPDF,
        estimateContentHeight,
        shouldUseChunking,
        splitContentIntoChunks,
        updateLoadingProgress
    };

})(document);

