/**
 * Header/Footer Module
 * Handles header and footer generation for PDF export
 */

(function (document) {
    'use strict';

    /**
     * Escapes HTML special characters to prevent XSS attacks
     * @param {string} text - The text to escape
     * @returns {string} - The escaped text safe for HTML insertion
     */
    const escapeHtml = (text) => {
        if (typeof text !== 'string') {
            return '';
        }
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#x27;',
            '/': '&#x2F;'
        };
        return text.replace(/[&<>"'/]/g, (char) => map[char]);
    };

    /**
     * Gets the topic title from h1 or page title
     * @returns {string} - The sanitized topic title
     */
    const getTopicTitle = () => {
        let title = '';
        const h1 = document.querySelector('h1');
        if (h1) {
            title = h1.textContent.trim();
        } else {
            title = document.title;
            title = title.replace(/\s*[-|]\s*(Documentation|Docs|ITRS).*$/i, '');
        }
        title = escapeHtml(title || 'Documentation');
        return title;
    };

    /**
     * Generates a filename from the page title, optionally prefixed with product name
     * @returns {string} - The cleaned filename
     */
    const generateFilename = () => {
        let productName = '';
        
        // Try to get product name from the search input filter element
        try {
            const inputFilter = document.querySelector('#inputFilter');
            if (inputFilter && inputFilter.hasAttribute('data-product')) {
                const productValue = inputFilter.getAttribute('data-product');
                if (productValue && productValue.trim()) {
                    // Clean product name for filename
                    productName = productValue.trim()
                        .replace(/[^a-zA-Z0-9\s\-]/g, '')
                        .replace(/\s+/g, '-')
                        .toLowerCase();
                }
            }
        } catch (error) {
            console.warn('[PDF] Could not extract product name:', error);
            // Continue without product name
        }
        
        // Get page title
        let title = document.title;
        title = title.replace(/\s*[-|]\s*(Documentation|Docs|ITRS|Geneos).*$/i, '');
        
        if (!title || title.toLowerCase().includes('documentation') || title.toLowerCase() === 'home') {
            const h1 = document.querySelector('h1');
            if (h1) {
                title = h1.textContent.trim();
            }
        }
        
        // Clean title for filename (no character limit)
        title = title.replace(/[^a-zA-Z0-9\s\-]/g, '')
                     .replace(/\s+/g, '-')
                     .toLowerCase();
        
        // Combine product name and title
        if (productName && title) {
            return `${productName}-${title}`;
        } else if (productName) {
            return productName;
        } else if (title) {
            return title;
        }
        
        return 'documentation';
    };

    /**
     * Loads logo image and converts to base64
     * @returns {Promise<Object|null>} - Logo data with base64, width, and height
     */
    const loadLogoAsBase64 = () => {
        return new Promise((resolve) => {
            const currentPath = window.location.pathname;
            const docsMatch = currentPath.match(/^(\/docs\/[^\/]+(?:\/([^\/]+))?)/);
            const possiblePaths = [];
            
            if (docsMatch) {
                const basePath = docsMatch[1];
                const versionOrProduct = docsMatch[2];
                const productMatch = basePath.match(/^\/docs\/([^\/]+)/);
                const productName = productMatch ? productMatch[1] : null;
                
                if (productName) {
                    if (versionOrProduct) {
                        possiblePaths.push(`/docs/${productName}/${versionOrProduct}/images/itrs-logo-pdf.png`);
                    }
                    possiblePaths.push(`/docs/all/images/itrs-logo-pdf.png`);
                }
            }
            
            let pathIndex = 0;
            
            const tryNextPath = () => {
                if (pathIndex >= possiblePaths.length) {
                    resolve(null);
                    return;
                }
                
                const logoPath = possiblePaths[pathIndex];
                pathIndex++;
                
                const img = new Image();
                img.crossOrigin = 'anonymous';
                
                img.onload = () => {
                    try {
                        const canvas = document.createElement('canvas');
                        canvas.width = img.width;
                        canvas.height = img.height;
                        const ctx = canvas.getContext('2d');
                        ctx.drawImage(img, 0, 0);
                        const base64 = canvas.toDataURL('image/png');
                        resolve({
                            base64: base64,
                            width: img.width,
                            height: img.height
                        });
                    } catch (error) {
                        tryNextPath();
                    }
                };
                
                img.onerror = () => {
                    tryNextPath();
                };
                
                img.src = logoPath;
            };
            
            tryNextPath();
        });
    };

    /**
     * Creates a header element for PDF
     * @param {string} topicTitle - The topic title
     * @param {Object} logoData - Logo data with base64
     * @returns {HTMLElement} - The header element
     */
    const createHeaderElement = (topicTitle, logoData) => {
        const headerDiv = document.createElement('div');
        headerDiv.className = 'pdf-header';
        headerDiv.style.cssText = `
            width: 100%;
            height: 60px;
            background: white;
            border-bottom: 2px solid #333;
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 54px;
            font-size: 12px;
            font-weight: bold;
            color: black;
            box-sizing: border-box;
            page-break-after: avoid;
            page-break-inside: avoid;
        `;
        
        const titleSpan = document.createElement('span');
        titleSpan.textContent = topicTitle;
        titleSpan.style.cssText = 'flex: 1; text-align: left;';
        headerDiv.appendChild(titleSpan);
        
        if (logoData && logoData.base64) {
            const logoImg = document.createElement('img');
            logoImg.src = logoData.base64;
            logoImg.style.cssText = 'max-height: 20px; max-width: 60px; object-fit: contain;';
            headerDiv.appendChild(logoImg);
        }
        
        return headerDiv;
    };

    /**
     * Creates a footer element for PDF
     * @param {number} pageNum - Current page number
     * @param {number} totalPages - Total number of pages
     * @returns {HTMLElement} - The footer element
     */
    const createFooterElement = (pageNum, totalPages) => {
        const footerDiv = document.createElement('div');
        footerDiv.className = 'pdf-footer';
        footerDiv.style.cssText = `
            width: 100%;
            height: 40px;
            background: white;
            border-top: 2px solid #333;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 0 54px;
            font-size: 10px;
            color: black;
            box-sizing: border-box;
            page-break-before: avoid;
            page-break-inside: avoid;
        `;
        footerDiv.textContent = `Page ${pageNum}`;
        return footerDiv;
    };

    /**
     * Gets header and footer HTML for html2pdf
     * @returns {Promise<Object>} - Object with headerHTML and footerHTML
     */
    const getHeaderFooterHTML = async () => {
        const topicTitle = getTopicTitle();
        const logoData = await loadLogoAsBase64();
        
        let headerHTML = `
            <div style="
                width: 100%;
                height: 60px;
                background: white;
                border-bottom: 2px solid #333;
                display: flex;
                align-items: center;
                justify-content: space-between;
                padding: 0 54px;
                font-size: 12px;
                font-weight: bold;
                color: black;
                box-sizing: border-box;
            ">
                <span style="flex: 1; text-align: left;">${topicTitle}</span>
        `;
        
        if (logoData && logoData.base64) {
            headerHTML += `
                <img src="${logoData.base64}" style="max-height: 20px; max-width: 60px; object-fit: contain;" />
            `;
        }
        
        headerHTML += '</div>';
        
        const footerHTML = `
            <div style="
                width: 100%;
                height: 40px;
                background: white;
                border-top: 2px solid #333;
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 0 54px;
                font-size: 10px;
                color: black;
                box-sizing: border-box;
            ">
                Page <span class="pageNumber"></span> of <span class="totalPages"></span>
            </div>
        `;
        
        return { headerHTML, footerHTML };
    };

    /**
     * Adds header and footer to PDF using jsPDF
     * @param {Object} pdf - The jsPDF instance
     * @param {string} topicTitle - The topic title
     * @param {Object} logoData - Logo data with base64
     */
    const addHeaderAndFooterInReservedSpace = (pdf, topicTitle, logoData) => {
        const pageWidthInches = pdf.internal.pageSize.getWidth();
        const pageHeightInches = pdf.internal.pageSize.getHeight();
        const sideMargin = 0.5;
        const topMargin = 1.4;
        const bottomMargin = 1.2;
        const totalPages = pdf.internal.getNumberOfPages();
        
        const fontList = pdf.getFontList();
        const hasBarlow = fontList && (fontList['Barlow'] || Object.keys(fontList).some(key => key.toLowerCase().includes('barlow')));
        const headerFont = hasBarlow ? 'Barlow' : 'helvetica';
        const headerFontStyle = 'bold';
        const footerFont = hasBarlow ? 'Barlow' : 'helvetica';
        const footerFontStyle = 'normal';
        
        for (let i = 1; i <= totalPages; i++) {
            pdf.setPage(i);
            pdf.saveGraphicsState();
            
            const opacity = 0.5;
            const grayValue = Math.round(255 * (1 - opacity));
            
            // Header
            const headerY = 0.5;
            const headerLineY = headerY + 0.25;
            pdf.setDrawColor(grayValue, grayValue, grayValue);
            pdf.setLineWidth(0.01);
            pdf.line(sideMargin, headerLineY, pageWidthInches - sideMargin, headerLineY);
            
            pdf.setFontSize(12);
            pdf.setFont(headerFont, headerFontStyle);
            pdf.setTextColor(grayValue, grayValue, grayValue);
            const titleX = sideMargin;
            pdf.text(topicTitle, titleX, headerY, { align: 'left' });
            
            if (logoData && logoData.base64) {
                try {
                    if (logoData.width && logoData.height && logoData.width > 0 && logoData.height > 0) {
                        const logoMaxWidthInches = 0.5;
                        const logoMaxHeightInches = 0.17;
                        const imgAspectRatio = logoData.width / logoData.height;
                        
                        if (isFinite(imgAspectRatio) && imgAspectRatio > 0) {
                            let logoWidthInches = logoMaxWidthInches;
                            let logoHeightInches = logoMaxWidthInches / imgAspectRatio;
                            
                            if (logoHeightInches > logoMaxHeightInches) {
                                logoHeightInches = logoMaxHeightInches;
                                logoWidthInches = logoMaxHeightInches * imgAspectRatio;
                            }
                            
                            if (logoWidthInches <= pageWidthInches && logoHeightInches <= pageHeightInches) {
                                const logoX = pageWidthInches - sideMargin - logoWidthInches;
                                const logoY = headerY - 0.125;
                                
                                if (logoX >= 0 && logoY >= 0 && logoX + logoWidthInches <= pageWidthInches && logoY + logoHeightInches <= pageHeightInches) {
                                    pdf.addImage(logoData.base64, 'PNG', logoX, logoY, logoWidthInches, logoHeightInches);
                                }
                            }
                        }
                    }
                } catch (error) {
                    // Could not add logo
                }
            }
            
            // Footer
            const footerY = pageHeightInches - 0.5;
            const footerLineY = footerY - 0.25;
            pdf.setDrawColor(grayValue, grayValue, grayValue);
            pdf.setLineWidth(0.01);
            pdf.line(sideMargin, footerLineY, pageWidthInches - sideMargin, footerLineY);
            
            pdf.setFontSize(10);
            pdf.setFont(footerFont, footerFontStyle);
            pdf.setTextColor(grayValue, grayValue, grayValue);
            const pageText = `Page ${i} of ${totalPages}`;
            const pageTextX = pageWidthInches / 2;
            pdf.text(pageText, pageTextX, footerY, { align: 'center' });
            
            pdf.restoreGraphicsState();
        }
    };

    // Export to global scope
    window.PDFHeaderFooter = {
        getTopicTitle,
        generateFilename,
        loadLogoAsBase64,
        createHeaderElement,
        createFooterElement,
        getHeaderFooterHTML,
        addHeaderAndFooterInReservedSpace,
        escapeHtml
    };

})(document);

