/**
 * Image Processing Module
 * Handles image preprocessing, CORS testing, and URL rewriting for PDF export
 */

(function (document) {
    'use strict';

    const rewriteToDocscdnPath = (url) => {
        try {
            // Resolve relative URLs to absolute
            const fullUrl = new URL(url, window.location.href);
            
            // If already pointing to docscdn, convert directly
            if (fullUrl.hostname === 'docscdn.itrsgroup.com' || fullUrl.hostname.endsWith('.docscdn.itrsgroup.com')) {
                // Extract the path after the domain
                // e.g., https://docscdn.itrsgroup.com/geneos/images/file.png -> /docscdn/geneos/images/file.png
                const cdnPath = fullUrl.pathname;
                const rewrittenUrl = `/docscdn${cdnPath}${fullUrl.search}${fullUrl.hash}`;
                return rewrittenUrl;
            }
            
            // If same-origin and might redirect to CDN, use pattern matching to construct /docscdn/ path
            const currentOrigin = window.location.origin;
            if (fullUrl.origin === currentOrigin) {
                const pathname = fullUrl.pathname;
                
                // Pattern 1: /docs/geneos/collection/(plugin)/(version)/(project-folder)/(images|videos)/(file)
                // Redirects to: $3/$4/$5
                // Example: /docs/geneos/collection/jdbc/4.0.0/geneos-jdbc/images/file.png -> /docscdn/geneos-jdbc/images/file.png
                const collectionMatch = pathname.match(/^\/docs\/geneos\/collection\/[^/]+\/[^/]+\/([^/]+)\/(images|videos)\/(.+)$/);
                if (collectionMatch) {
                    const projectFolder = collectionMatch[1];
                    const mediaType = collectionMatch[2];
                    const filePath = collectionMatch[3];
                    const rewrittenUrl = `/docscdn/${projectFolder}/${mediaType}/${filePath}${fullUrl.search}${fullUrl.hash}`;
                    return rewrittenUrl;
                }
                
                // Pattern 2: /docs/itrs-analytics/platform/(version)/(project-folder)/(images|videos)/(file)
                // Redirects to: $2/$3/$4
                const platformMatch = pathname.match(/^\/docs\/itrs-analytics\/platform\/[^/]+\/([^/]+)\/(images|videos)\/(.+)$/);
                if (platformMatch) {
                    const projectFolder = platformMatch[1];
                    const mediaType = platformMatch[2];
                    const filePath = platformMatch[3];
                    const rewrittenUrl = `/docscdn/${projectFolder}/${mediaType}/${filePath}${fullUrl.search}${fullUrl.hash}`;
                    return rewrittenUrl;
                }
                
                // Pattern 3: /docs/obcerv/platform/(version)/(project-folder)/(images|videos)/(file)
                // Redirects to: $2/$3/$4
                const obcervPlatformMatch = pathname.match(/^\/docs\/obcerv\/platform\/[^/]+\/([^/]+)\/(images|videos)\/(.+)$/);
                if (obcervPlatformMatch) {
                    const projectFolder = obcervPlatformMatch[1];
                    const mediaType = obcervPlatformMatch[2];
                    const filePath = obcervPlatformMatch[3];
                    const rewrittenUrl = `/docscdn/${projectFolder}/${mediaType}/${filePath}${fullUrl.search}${fullUrl.hash}`;
                    return rewrittenUrl;
                }
                
                // Pattern 4: /docs/itrs-analytics/app/(version)/(project-folder)/(images|videos)/(file)
                // Redirects to: $3/$4/$5
                const appMatch = pathname.match(/^\/docs\/itrs-analytics\/app\/[^/]+\/[^/]+\/([^/]+)\/(images|videos)\/(.+)$/);
                if (appMatch) {
                    const projectFolder = appMatch[1];
                    const mediaType = appMatch[2];
                    const filePath = appMatch[3];
                    const rewrittenUrl = `/docscdn/${projectFolder}/${mediaType}/${filePath}${fullUrl.search}${fullUrl.hash}`;
                    return rewrittenUrl;
                }
                
                // Pattern 5: /docs/obcerv/app/(version)/(project-folder)/(images|videos)/(file)
                // Redirects to: $3/$4/$5
                const obcervAppMatch = pathname.match(/^\/docs\/obcerv\/app\/[^/]+\/[^/]+\/([^/]+)\/(images|videos)\/(.+)$/);
                if (obcervAppMatch) {
                    const projectFolder = obcervAppMatch[1];
                    const mediaType = obcervAppMatch[2];
                    const filePath = obcervAppMatch[3];
                    const rewrittenUrl = `/docscdn/${projectFolder}/${mediaType}/${filePath}${fullUrl.search}${fullUrl.hash}`;
                    return rewrittenUrl;
                }
                
                // Pattern 6: /docs/(all)/global/(images|videos)/(file)
                // Redirects to: $1/$2/$3
                const globalMatch = pathname.match(/^\/docs\/(all)\/global\/(images|videos)\/(.+)$/);
                if (globalMatch) {
                    const all = globalMatch[1];
                    const mediaType = globalMatch[2];
                    const filePath = globalMatch[3];
                    const rewrittenUrl = `/docscdn/${all}/${mediaType}/${filePath}${fullUrl.search}${fullUrl.hash}`;
                    return rewrittenUrl;
                }
                
                // Pattern 7: /docs/(app)/(app)/(images|videos)/(file) - same app name twice
                // Redirects to: $app/$2/$3
                const sameAppMatch = pathname.match(/^\/docs\/([^/]+)\/\1\/(images|videos)\/(.+)$/);
                if (sameAppMatch) {
                    const app = sameAppMatch[1];
                    const mediaType = sameAppMatch[2];
                    const filePath = sameAppMatch[3];
                    const rewrittenUrl = `/docscdn/${app}/${mediaType}/${filePath}${fullUrl.search}${fullUrl.hash}`;
                    return rewrittenUrl;
                }
                
                // Pattern 8: /docs/(project)/(version)/(project-folder)/(images|videos)/(file) - default pattern
                // Redirects to: $3/$4/$5
                // This must be checked last as it's the most general pattern
                const defaultMatch = pathname.match(/^\/docs\/[^/]+\/[^/]+\/([^/]+)\/(images|videos)\/(.+)$/);
                if (defaultMatch && !pathname.includes('/platform/') && !pathname.includes('/app/') && !pathname.includes('/collection/')) {
                    const projectFolder = defaultMatch[1];
                    const mediaType = defaultMatch[2];
                    const filePath = defaultMatch[3];
                    const rewrittenUrl = `/docscdn/${projectFolder}/${mediaType}/${filePath}${fullUrl.search}${fullUrl.hash}`;
                    return rewrittenUrl;
                }
            }
        } catch (e) {
            console.warn('[PDF] Failed to parse URL for /docscdn/ rewriting:', url, e);
        }
        return null;
    };

    const testImageLoad = (imageUrl) => {
        // Test if image can be loaded with CORS (quick check before PDF generation)
        return new Promise((resolve) => {
            const testImg = new Image();
            testImg.crossOrigin = 'anonymous';
            
            const timeout = setTimeout(() => {
                console.error('[PDF] Image CORS test timeout for:', imageUrl);
                console.error('[PDF] This usually indicates a network issue or the image server is not responding');
                resolve(false);
            }, 3000); // 3 second timeout for quick check
            
            testImg.onload = () => {
                clearTimeout(timeout);
                resolve(true);
            };
            
            testImg.onerror = (error) => {
                clearTimeout(timeout);
                // Log detailed CORS error information
                console.error('[PDF] Image CORS test failed for:', imageUrl);
                console.error('[PDF] CORS Error Details:', {
                    imageUrl: imageUrl,
                    error: error,
                    errorType: error?.type || 'unknown',
                    errorTarget: error?.target || null,
                    currentOrigin: window.location.origin,
                    imageOrigin: (() => {
                        try {
                            return new URL(imageUrl).origin;
                        } catch (e) {
                            return 'unknown';
                        }
                    })(),
                    crossOrigin: testImg.crossOrigin,
                    message: 'Image failed to load due to CORS policy. The server may not be sending the required Access-Control-Allow-Origin header.'
                });
                
                // Try to get more specific error information if available
                if (error?.target) {
                    const imgElement = error.target;
                    console.error('[PDF] Image element details:', {
                        src: imgElement.src,
                        complete: imgElement.complete,
                        naturalWidth: imgElement.naturalWidth,
                        naturalHeight: imgElement.naturalHeight
                    });
                }
                
                resolve(false);
            };
            
            testImg.src = imageUrl;
        });
    };
    
    const replaceImageWithLink = (img, fullUrl) => {
        // Replace image with link button (fallback when CORS prevents loading)
        const doc = img.ownerDocument || document;
        
        const imageContainer = doc.createElement('div');
        const imageLink = doc.createElement('a');
        
        imageLink.href = fullUrl;
        imageLink.textContent = 'Click to view image';
        imageLink.target = '_blank';
        imageLink.style.padding = '10px';
        imageLink.style.backgroundColor = '#070032';
        imageLink.style.borderRadius = '6px';
        imageLink.style.color = 'white';
        imageLink.style.display = 'inline-block';
        imageLink.style.margin = '10px 0';
        imageLink.style.textDecoration = 'none';
        
        imageContainer.style.width = '100%';
        imageContainer.style.display = 'flex';
        imageContainer.style.alignItems = 'center';
        imageContainer.style.justifyContent = 'center';
        imageContainer.style.margin = '20px 0';
        
        imageContainer.appendChild(imageLink);
        
        if (img.parentNode) {
            img.parentNode.replaceChild(imageContainer, img);
        } else {
            console.error('[PDF] Cannot replace image - no parentNode');
        }
    };

    const preprocessImagesForPDF = async (contentClone) => {
        // Try to render images in PDF first, only replace with links if CORS prevents loading
        const images = contentClone.querySelectorAll('img');
        const currentOrigin = window.location.origin;
        const knownCDNDomains = ['docscdn.itrsgroup.com', 'cdn.itrsgroup.com'];
        
        const imagePromises = Array.from(images).map(async (img) => {
            // Skip GIFs (already handled)
            if (img.src && img.src.includes('.gif')) {
                return;
            }
            
            const srcAttr = img.getAttribute('src');
            
            // Skip data URIs and images without src
            if (!srcAttr || srcAttr.startsWith('data:')) {
                return;
            }
            
            // Check if image is from external origin or CDN
            // Also check if it might redirect to CDN (relative paths with /images/ often redirect)
            let isExternal = false;
            let mightRedirectToCDN = false;
            
            try {
                const imgUrl = new URL(srcAttr, window.location.href);
                isExternal = imgUrl.origin !== currentOrigin;
                const isCDN = knownCDNDomains.some(domain => imgUrl.hostname.includes(domain));
                
                // Check if image might redirect to CDN
                // Any path with /images/ that's not a data URI might redirect
                // Also check for common patterns that redirect
                mightRedirectToCDN = srcAttr.includes('/images/') && !srcAttr.startsWith('data:');
                
                isExternal = isExternal || isCDN || mightRedirectToCDN;
            } catch (e) {
                // If URL parsing fails and it's a relative path with /images/, assume it might redirect
                if (srcAttr.includes('/images/') && !srcAttr.startsWith('data:')) {
                    isExternal = true;
                    mightRedirectToCDN = true;
                }
            }
            
            // For external images OR images that might redirect to CDN, try rewriting to /docscdn/ path
            // This allows us to use the proxy which keeps requests same-origin (no CORS needed)
            if (isExternal || mightRedirectToCDN) {
                // Get full URL first
                let fullUrl = srcAttr;
                if (!srcAttr.startsWith('http://') && !srcAttr.startsWith('https://')) {
                    try {
                        fullUrl = new URL(srcAttr, window.location.href).href;
                    } catch (e) {
                        fullUrl = window.location.origin + (srcAttr.startsWith('/') ? '' : '/') + srcAttr;
                    }
                }
                
                // Try to rewrite to /docscdn/ path (uses pattern matching to construct path)
                const rewrittenUrl = rewriteToDocscdnPath(fullUrl);
                
                if (rewrittenUrl) {
                    // Test if the rewritten URL actually loads (it might fail on dev/uat due to environment prefixes)
                    try {
                        const rewrittenCanLoad = await testImageLoad(rewrittenUrl);
                        
                        if (rewrittenCanLoad) {
                            // Rewritten URL works - use it (same-origin, no CORS needed)
                            img.setAttribute('src', rewrittenUrl);
                            img.src = rewrittenUrl;
                            // Set crossOrigin for html2pdf/html2canvas (even though same-origin, it's good practice)
                            img.setAttribute('crossOrigin', 'anonymous');
                            img.crossOrigin = 'anonymous';
                        } else {
                            // Rewritten URL doesn't load - fall back to testing original URL with CORS
                            console.warn('[PDF] Rewritten /docscdn/ URL failed to load, testing original URL with CORS:', fullUrl);
                            try {
                                const canLoad = await testImageLoad(fullUrl);
                                
                                if (!canLoad) {
                                    // CORS prevents loading - replace with link button
                                    console.warn('[PDF] CORS blocked image, replacing with link button for:', fullUrl);
                                    replaceImageWithLink(img, fullUrl);
                                } else {
                                    // Original URL can load - set crossOrigin for html2pdf/html2canvas
                                    img.setAttribute('crossOrigin', 'anonymous');
                                    img.crossOrigin = 'anonymous';
                                }
                            } catch (error) {
                                // Catch any errors during CORS testing
                                console.error('[PDF] Exception during CORS test for:', fullUrl, error);
                                // Replace with link as fallback
                                replaceImageWithLink(img, fullUrl);
                            }
                        }
                    } catch (error) {
                        // Error testing rewritten URL - fall back to original URL
                        console.error('[PDF] Exception testing rewritten URL:', rewrittenUrl, error);
                        try {
                            const canLoad = await testImageLoad(fullUrl);
                            
                            if (!canLoad) {
                                // CORS prevents loading - replace with link button
                                console.warn('[PDF] CORS blocked image, replacing with link button for:', fullUrl);
                                replaceImageWithLink(img, fullUrl);
                            } else {
                                // Image can load - set crossOrigin for html2pdf/html2canvas
                                img.setAttribute('crossOrigin', 'anonymous');
                                img.crossOrigin = 'anonymous';
                            }
                        } catch (error2) {
                            // Both failed - replace with link
                            console.error('[PDF] Both rewritten and original URLs failed, replacing with link button');
                            replaceImageWithLink(img, fullUrl);
                        }
                    }
                } else {
                    // Could not rewrite - test if original URL works with CORS
                    try {
                        const canLoad = await testImageLoad(fullUrl);
                        
                        if (!canLoad) {
                            // CORS prevents loading - replace with link button
                            console.warn('[PDF] CORS blocked image, replacing with link button for:', fullUrl);
                            replaceImageWithLink(img, fullUrl);
                        } else {
                            // Image can load - set crossOrigin for html2pdf/html2canvas
                            img.setAttribute('crossOrigin', 'anonymous');
                            img.crossOrigin = 'anonymous';
                        }
                    } catch (error) {
                        // Catch any errors during CORS testing
                        console.error('[PDF] Exception during CORS test for:', fullUrl, error);
                        // Replace with link as fallback
                        replaceImageWithLink(img, fullUrl);
                    }
                }
            } else {
                // Same-origin images - set crossOrigin for html2pdf/html2canvas
                img.setAttribute('crossOrigin', 'anonymous');
                img.crossOrigin = 'anonymous';
            }
        });
        
        // Wait for all image checks to complete
        await Promise.all(imagePromises);
    };

    // Export to global scope
    window.PDFImageProcessing = {
        preprocessImagesForPDF,
        testImageLoad,
        replaceImageWithLink,
        rewriteToDocscdnPath
    };

})(document);

