/**
 * PDF Styling Module
 * Applies styling optimizations for PDF export
 */

(function (document) {
    'use strict';

    /**
     * Applies simple styling optimizations for PDF export
     * @param {Node} contentClone - The content clone to style
     */
    const addSimplePdfStyling = (contentClone) => {
        // Remove copy and wrap buttons that interfere with PDF
        const codeButtons = contentClone.querySelectorAll('.copy-code-button, .wrap-code-button');
        codeButtons.forEach(button => {
            if (button.parentNode) {
                button.parentNode.removeChild(button);
            }
        });

        // Style headings for PDF
        const headings = contentClone.querySelectorAll('h1, h2, h3, h4, h5, h6');
        // Navy color matching @color-navy (#070032)
        const headingColor = '#070032';
        
        headings.forEach(heading => {
            const tagName = heading.tagName.toLowerCase();
            // Apply inline styles based on heading level (matching standards.less)
            // Set color for all headings (html2canvas needs inline styles)
            heading.style.color = headingColor;
            
            switch(tagName) {
                case 'h1':
                    heading.style.fontSize = '4rem';
                    heading.style.fontWeight = '700';
                    heading.style.lineHeight = '4.4rem';
                    heading.style.marginTop = '2.5rem';
                    heading.style.marginBottom = '1.5rem';
                    break;
                case 'h2':
                    heading.style.fontSize = '2.75rem';
                    heading.style.fontWeight = '700';
                    heading.style.lineHeight = '3.08rem';
                    heading.style.marginTop = '2.5rem';
                    heading.style.marginBottom = '1.5rem';
                    break;
                case 'h3':
                    heading.style.fontSize = '1.5rem';
                    heading.style.fontWeight = '600';
                    heading.style.lineHeight = '1.8rem';
                    heading.style.marginTop = '2.5rem';
                    heading.style.marginBottom = '1.5rem';
                    break;
                case 'h4':
                    heading.style.fontSize = '1.25rem';
                    heading.style.fontWeight = '600';
                    heading.style.lineHeight = '1.5rem';
                    heading.style.marginTop = '2.5rem';
                    heading.style.marginBottom = '1.5rem';
                    break;
                case 'h5':
                    heading.style.fontSize = '1rem';
                    heading.style.fontWeight = '600';
                    heading.style.lineHeight = '1.3rem';
                    heading.style.letterSpacing = '0.05rem';
                    heading.style.marginTop = '2.5rem';
                    heading.style.marginBottom = '1.5rem';
                    break;
                case 'h6':
                    heading.style.fontSize = '0.75rem';
                    heading.style.fontWeight = '600';
                    heading.style.lineHeight = '1rem';
                    heading.style.letterSpacing = '0.05rem';
                    heading.style.marginTop = '2.5rem';
                    heading.style.marginBottom = '1.5rem';
                    break;
            }
            // Reset first heading margin
            if (heading === contentClone.querySelector('h1, h2, h3, h4, h5, h6')) {
                heading.style.marginTop = '0';
            }
        });

        // Style the main code containers
        const codeContainers = contentClone.querySelectorAll('.highlight, pre');
        codeContainers.forEach(container => {
            container.style.backgroundColor = '#F8F8F8';
            container.style.borderRadius = '8px';
            container.style.color = '#333333';
            container.style.padding = '12px';
            container.style.marginBottom = '1em';
            container.style.overflow = 'visible'; // Prevent cutting off content
        });

        // Handle chroma line number tables specifically
        const chromaTables = contentClone.querySelectorAll('.chroma .lntable');
        chromaTables.forEach(table => {
            table.style.width = '100%';
            table.style.borderSpacing = '0';
            table.style.borderCollapse = 'collapse';
            table.style.tableLayout = 'auto';
        });

        // Style block-level code content with proper monospace font and wrapping
        const codeElements = contentClone.querySelectorAll('.chroma code, pre code, code');
        codeElements.forEach(code => {
            // Check if this is truly a block-level code element
            const isInPreBlock = code.closest('pre') !== null;
            const isInChroma = code.closest('.chroma') !== null;
            const isMultiLine = code.textContent && code.textContent.includes('\n');
            
            // Skip inline codes - they'll be handled by the inline code section below
            const isInParagraph = code.closest('p') !== null;
            const isInListItem = code.closest('li') !== null;
            const isInTableCell = code.closest('td, th') !== null;
            const isInlineCode = (isInParagraph || isInListItem || isInTableCell) && !isInPreBlock && !isInChroma;
            
            if (isInlineCode) {
                return; // Skip, will be handled by inline code styling
            }
            
            // Apply block code styling
            code.style.fontFamily = "'Source Code Pro', 'Courier New', monospace";
            code.style.fontSize = '1.143rem';
            code.style.whiteSpace = 'pre-wrap'; // Preserve formatting but allow wrapping
            code.style.wordBreak = 'break-word'; // Break long words if needed
            code.style.overflowWrap = 'break-word'; // Additional word breaking
                code.style.display = 'block';
            code.style.padding = '0';
            code.style.margin = '0';
            code.style.backgroundColor = '#f1f3f4';
        });

        // Apply wrap-code-text style to all code blocks for better PDF rendering
        const chromaCodes = contentClone.querySelectorAll('.chroma > code');
        chromaCodes.forEach(code => {
            code.classList.add('wrap-code-text');
        });

        // Handle inline code differently - must stay inline with surrounding text
        const inlineCodes = contentClone.querySelectorAll('p code, li code, td code, th code');
        inlineCodes.forEach(code => {
            // Skip if this code is inside a pre or chroma block (those are handled separately)
            if (code.closest('pre') || code.closest('.chroma')) {
                return;
            }
            code.style.display = 'inline-block'; // inline-block for better html2canvas rendering
            code.style.verticalAlign = 'middle'; // Align with text bottom for better positioning
            code.style.backgroundColor = '#f1f3f4';
            code.style.padding = '2px 4px';
            code.style.borderRadius = '3px';
            code.style.fontSize = '0.9em';
            code.style.fontFamily = "'Source Code Pro', 'Courier New', monospace";
            code.style.whiteSpace = 'normal'; // Use normal to collapse whitespace and flow with text
            code.style.wordBreak = 'break-word'; // Break long words at container boundary
            code.style.overflowWrap = 'break-word'; // Additional word breaking support
            code.style.lineHeight = 'inherit'; // Inherit line height from parent paragraph
        });
        
        // Disable ::before and ::after pseudo-elements in blockquotes for PDF
        // html2pdf/html2canvas doesn't render CSS pseudo-elements, so we disable them to prevent white space
        const style = document.createElement('style');
        style.textContent = `
            blockquote p::before, q p::before {
                content: none !important;
            }
            blockquote p::after, q p::after {
                content: none !important;
            }
        `;
        contentClone.appendChild(style);

    };

    // Export to global scope
    window.PDFStyling = {
        addSimplePdfStyling
    };

})(document);

