/**
 * SVG Processing Module
 * Converts SVG elements using <use> references to inline SVGs for PDF export.
 * html2canvas cannot resolve <use xlink:href> references, so we need to inline the SVG content.
 */

(function (document) {
    'use strict';

    /**
     * Converts SVG elements using <use> references to inline SVGs for PDF export.
     * @param {Node} contentClone - The cloned content node to process
     */
    const convertSvgUseToInline = (contentClone) => {
        // Find all SVG elements that use <use> references
        const svgsWithUse = contentClone.querySelectorAll('svg use');
        
        // Get the SVG sprite container (should be in the original document, not the clone)
        // We need to access the original document to find the sprite definitions
        const originalSvgSprite = document.querySelector('svg.svg-sprite');
        if (!originalSvgSprite) {
            console.warn('[PDF] SVG sprite not found, cannot convert SVG icons to inline');
            return;
        }

        svgsWithUse.forEach((useElement) => {
            const parentSvg = useElement.closest('svg');
            if (!parentSvg || !parentSvg.parentNode) {
                return;
            }

            // Get the icon ID from xlink:href or href attribute
            const href = useElement.getAttribute('xlink:href') || useElement.getAttribute('href');
            if (!href) {
                return;
            }

            // Extract icon ID (remove the # prefix if present)
            const iconId = href.replace('#', '');
            
            // Find the symbol in the original SVG sprite
            const symbol = originalSvgSprite.querySelector(`symbol#${iconId}`);
            if (!symbol) {
                console.warn(`[PDF] Symbol not found for icon: ${iconId}`);
                return;
            }

            // Get the viewBox from the symbol (or use parent SVG's viewBox)
            const viewBox = symbol.getAttribute('viewBox') || parentSvg.getAttribute('viewBox') || '0 0 24 24';
            
            // Parse viewBox to extract dimensions if needed
            const viewBoxValues = viewBox.split(/\s+/);
            const svgWidth = viewBoxValues.length >= 3 ? viewBoxValues[2] : '24';
            const svgHeight = viewBoxValues.length >= 4 ? viewBoxValues[3] : '24';
            
            // Get all classes from the original SVG
            const classes = parentSvg.getAttribute('class') || '';
            
            // Get the document context from the contentClone (important for proper node creation)
            const doc = contentClone.ownerDocument || document;
            
            // Create a new inline SVG in the correct document context
            const inlineSvg = doc.createElementNS('http://www.w3.org/2000/svg', 'svg');
            inlineSvg.setAttribute('class', classes);
            inlineSvg.setAttribute('viewBox', viewBox);
            inlineSvg.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
            
            // Set explicit width and height if not already present (html2canvas may need these)
            // Use pixels based on viewBox dimensions
            if (!parentSvg.hasAttribute('width')) {
                inlineSvg.setAttribute('width', `${svgWidth}px`);
            }
            if (!parentSvg.hasAttribute('height')) {
                inlineSvg.setAttribute('height', `${svgHeight}px`);
            }
            
            // Copy any other attributes from the original SVG (like width, height, style)
            Array.from(parentSvg.attributes).forEach(attr => {
                if (!['class', 'viewBox', 'xmlns'].includes(attr.name)) {
                    inlineSvg.setAttribute(attr.name, attr.value);
                }
            });
            
            // Import and append all children from the symbol into the new document context
            // Using importNode ensures proper namespace handling when moving between documents
            let hasContent = false;
            Array.from(symbol.children).forEach(child => {
                const importedChild = doc.importNode(child, true);
                inlineSvg.appendChild(importedChild);
                hasContent = true;
            });
            
            if (!hasContent) {
                console.warn(`[PDF] Symbol ${iconId} has no children to copy`);
            }
            
            // Get computed fill color from the original SVG (if available)
            // html2canvas doesn't handle 'currentColor' well, so we need explicit colors
            let computedFill = '#000000'; // default to black
            try {
                // Try to get the computed color from the original SVG in the main document
                const originalSvgInDoc = document.querySelector(`svg.icon.${iconId}`);
                if (originalSvgInDoc) {
                    const computedStyle = window.getComputedStyle(originalSvgInDoc);
                    const fillColor = computedStyle.fill || computedStyle.color;
                    // Convert 'currentColor' or other CSS values to a hex color if possible
                    if (fillColor && fillColor !== 'currentColor' && fillColor !== 'inherit') {
                        computedFill = fillColor;
                    }
                }
            } catch (e) {
                // Fallback to black if computed style fails
                console.warn(`[PDF] Could not get computed fill for ${iconId}, using black`);
            }
            
            // Ensure paths have fill attributes if they don't already have them
            // html2canvas may not apply CSS fill colors, so we need explicit attributes
            const paths = inlineSvg.querySelectorAll('path');
            paths.forEach(path => {
                const currentFill = path.getAttribute('fill');
                // If fill is missing, set it to the computed color
                // Preserve 'none' for intentionally transparent paths
                if (!currentFill) {
                    path.setAttribute('fill', computedFill);
                }
                // Note: We preserve 'none' as-is for transparent elements
            });
            
            // Ensure SVG has proper styling
            if (!inlineSvg.getAttribute('style')) {
                inlineSvg.setAttribute('style', 'display: inline-block;');
            }
            
            // Replace the original SVG with the inline version
            // Check parentNode again in case it was removed during processing
            if (!parentSvg.parentNode) {
                console.warn(`[PDF] Parent node no longer exists for SVG icon: ${iconId}`);
                return;
            }
            
            parentSvg.parentNode.replaceChild(inlineSvg, parentSvg);
            
        });
    };

    // Export to global scope
    window.PDFSvgProcessing = {
        convertSvgUseToInline
    };

})(document);

