(function (document) {

    // Initialize the script once the DOM is ready
    let init = () => {
            if (document.readyState !== 'loading') {
                eventHandler(); // If DOM already loaded
            } else {
                document.addEventListener('DOMContentLoaded', eventHandler); // Wait for DOMContentLoaded
            }
        },

        // Main logic for version picker setup
        eventHandler = async () => {
                let vp = document.querySelector('.version-picker'); // Get the version picker container

                // Exit early if the project doesn't have a version picker
                if (!vp) {
                    return;
                }

                let product = vp.dataset.product; // Product name from data attribute
                let versionListUrl = '';
                let select = vp.querySelector('select'); // The dropdown element

                // Construct URL to the product's version list
                if (product !== undefined) {
                    versionListUrl = '/versions/' + product + '.json';
                }

                // If no URL to fetch from, exit
                if (!versionListUrl) {
                    return;
                }
            
            // Request the version list via fetch API
            await requestVersionList(versionListUrl, select, vp, product);

            },

            requestVersionList = async (versionListUrl, select, vp, product) => {
                    try {
                        const res = await fetch(versionListUrl);

                        if (!res.ok) throw new Error();
                        const data = await res.json();
                        addVpOptions(data, select);
                        select.addEventListener('change', openVersion);
                        vp.addEventListener('mouseover', disableVersions(vp));
                        select.classList.remove('disabled');
                    
                    } catch (error) {
                        // Mark dropdown as disabled on failure
                        select.classList.add('disabled');
                        console.log(`Error on response while loading version picker for ${product}.`);
                    }
                },

                // Navigate to the selected version's URL
                openVersion = (event) => {
                    let optionValue = event.target.value;
                    if (optionValue != '') {
                        window.location.href = optionValue;
                    }
                },

                // Disable dropdown options if their URLs do not exist
                disableVersions = async (vp) => {
                    // Skip if already processed
                    if (vp.classList.contains('processed')) {
                        return;
                    }

                    vp.classList.add('processed');
                    let list = vp.querySelectorAll('option');

                    for (const option of list){
                        let url = option.value;
                        let last = url.substr(-1);
                        if (last !== 'l' && last !== 'm' && last !== '/') {
                            url += '/'; // Ensure trailing slash
                        }
                        await urlExists(url, option); // Check if the version URL exists
                    }
                },

                // Populate the version dropdown with options
                addVpOptions = (versions, select) => {
                    if (!Array.isArray(versions)) {
                        return;
                    }

                    const versionPattern = /\/\d*(\.\d*)+\//; // e.g. /1.2/ or /2.0.1/
                    let currentUrl = window.location.pathname;
                    let ver;
                    let currentVersion;

                    // Extract the current version from the URL
                    if ((ver = versionPattern.exec(currentUrl)) !== null) {
                        currentVersion = ver[0].slice(0, -1); // Remove trailing slash
                        let endPage = currentUrl.split(currentVersion)[1]; // Get the rest of the path

                        // Add each version as an <option>
                        versions.forEach((version, i)=>{
                            select[select.options.length] = new Option(
                                version.label,
                                version.path + endPage,
                                version.path.indexOf(currentVersion) > -1, // isSelected
                                version.path.indexOf(currentVersion) > -1 // isDefaultSelected
                            );
                        })
                    }
                },

                // Check if a URL exists using a HEAD request and disable the option if not
                urlExists = async (url, option) => {
                    fetch(url, {method: 'HEAD'}).then((res)=>{
                        if (res.status >= 200 && res.status < 400) {
                            option.removeAttribute('disabled'); // URL works
                        } else {
                            option.setAttribute('disabled', 'disabled'); // URL doesn't work
                        }
                    })

                };

    // Start the process
    init();

})(document);