﻿// JS for tracking scroll on sidebar TOC
(function(document) {
    'use strict';
  
    // Initialize when DOM is ready
    let init = function() {
      if (document.readyState !== 'loading') {
        eventHandler(); // DOM is already ready
      } else {
        document.addEventListener('DOMContentLoaded', eventHandler); // Wait for DOM to load
      }
    },
  
    eventHandler = function() {
      const toc = document.querySelector('.toc-navigation');
  
      if (!toc)
        return; // No TOC found, exit
  
      // Intersection Observer settings:
      let observerOptions = {
        rootMargin: '-100px 0px 0px 0px', // Start detecting slightly before anchor hits top
        threshold: 1                      // Only when fully visible
      };
  
      let observer = new IntersectionObserver(observerCallback, observerOptions);
  
      // Observer callback toggles a visibility class on anchors
      function observerCallback(entries, observer) {
        entries.forEach(entry => {
          entry.target.classList.toggle(
            'anchor-toc-visible',
            entry.isIntersecting
          );
        });
  
        setActiveTOC(); // Determine which TOC item should be active
      };
  
      // Observe all anchor targets referenced in TOC
      let target = '.toc-navigation a[href]';
  
      document.querySelectorAll(target).forEach((i) => {
        let targetAnchor = document.querySelector(
          '.algolia-content .anchor-bookmark[href="' + i.getAttribute('href') + '"]'
        );
  
        if (!targetAnchor) return;
  
        targetAnchor.classList.add('anchor-bookmark-toc');
        observer.observe(targetAnchor); // Start observing the anchor in the content
      });
    };
  
    // Figure out which TOC item is closest to the center of the viewport
    function setActiveTOC() {
      let visibleAnchors = document.querySelectorAll('.anchor-toc-visible');
      let thereAreVisibleAnchors = (visibleAnchors && visibleAnchors.length > 0);
  
      // If none are visible, fallback to observing all known bookmarks
      if (!thereAreVisibleAnchors) {
        visibleAnchors = document.querySelectorAll('.anchor-bookmark-toc');
      }
  
      if (!visibleAnchors || !visibleAnchors.length) return;
  
      // Calculate vertical center point, slightly below the real center
      let header = document.querySelector('#header-placeholder').offsetHeight;
      let verticalCenter = (window.innerHeight - header) / 3 + header;
  
      let closerTop = null;
      let closer = null;
  
      // Find the anchor closest to verticalCenter
      visibleAnchors.forEach((i, key) => {
        let currentTop = i.getBoundingClientRect().top;
  
        // First iteration
        if (closerTop === null) {
          closerTop = currentTop;
          closer = i;
        }
  
        let closerThanPrevious = Math.abs(verticalCenter - closerTop) >= Math.abs(verticalCenter - currentTop);
  
        if (
          (!thereAreVisibleAnchors && currentTop < window.innerHeight) || // fallback mode
          (thereAreVisibleAnchors && closerThanPrevious) ||               // better match
          (currentTop < verticalCenter && key === visibleAnchors.length - 1) // last, and it passed midpoint
        ) {
          closer = i;
          closerTop = currentTop;
        }
      });
  
      if (!closer) return;
  
      activateTOC(closer);
    };
  
    // Highlight the active TOC link corresponding to the visible anchor
    function activateTOC(anchor) {
      let anchorRef = anchor.getAttribute('href');
      let tocAnchor = document.querySelector('.toc-navigation [href="' + anchorRef + '"]');
  
      if (!tocAnchor) return; // No matching TOC link found
  
      cleanTOC(); // Clear any previous active links
  
      tocAnchor.classList.add('li-active'); // Highlight the current one
  
      // Keep the active item in view (especially useful for long TOCs)
      tocAnchor.scrollIntoView({ block: "nearest", inline: "nearest" });
    };
  
    // Remove active class from all TOC links
    function cleanTOC() {
      let tocAnchorLis = document.querySelectorAll('.toc-navigation a');
      tocAnchorLis.forEach((i) => {
        i.classList.remove('li-active');
      });
    };
  
    // Start the process
    init();
  
  })(document);
  