const getLatestVersion = () => {
    return new Promise((resolve) => {
        // Select the version picker element from the DOM
        let vp = document.querySelector('.version-picker');
        // Construct the full current URL
        let fullURL = location.protocol + '//' + location.host + location.pathname;

        // If no version picker is found, resolve with the current URL as the canonical one
        // This ensures the canonical url is set on unversioned products and global-docs
        if (!vp) {
            resolve(fullURL);
            return;
        }

        // Retrieve the product name from the version picker dataset
        let product = vp.dataset.product;

        // validation: product should be a string containing lowercase letters, numbers (op5), and may also include '-' and '_'
        if (typeof product !== 'string' || !/^[a-z0-9_-]+$/.test(product) || product === undefined) {
            reject(new Error('Failed to retrieve product'));
            return;

        }

        // Construct the URL for the version list JSON file
        let versionListUrl = `/versions/${product}.json`;

        // Fetch the version list JSON
        fetch(versionListUrl)
            .then(response => {
                // If the response is not successful, throw an error
                if (!response.ok) throw new Error(`Error loading version picker for ${product}.`);
                return response.json(); // Parse the response as JSON
            })
            .then(data => {
                // Extract the first path from the version list (if available)
                const str = data[0]?.path;
                // Match the version number from the path (e.g., "7.1.0")
                const versionMatch = str?.match(/(\d+\.\d+\.\d+)/);
                // If no version is found, throw an error
                if (!versionMatch) throw new Error('Version not found in path.');

                // Extract the version string
                const version = versionMatch[0];
                // Replace the existing version in the URL with the latest version
                resolve(fullURL.replace(/(\d+\.\d+\.\d+)/, version));
            })
            .catch(error => {
                // Log any errors and resolve as undefined
                console.error(error.message);
                resolve(undefined);
            });
    });
};

// Call the function and update the canonical link if a new version URL is found
getLatestVersion().then((updatedUrl) => {
    if (updatedUrl) {
        // Try to find an existing canonical link, otherwise create a new one
        let canonicalLink = document.querySelector("link[rel='canonical']") || document.createElement('link');
        canonicalLink.setAttribute('rel', 'canonical'); // Set the rel attribute
        canonicalLink.setAttribute('href', updatedUrl); // Set the href to the updated URL

        // If the canonical link is new, append it to the document head
        if (!document.head.contains(canonicalLink)) {
            document.head.appendChild(canonicalLink);
        }
    }
});
