// Global helper functions and config.

const globalConfig = {
  // Base URLs.
  baseUrl: {
    // Connector
    connector: {
      loc: 'https://locconnector.itrsgroup.com',
      dev: 'https://devconnector.itrsgroup.com',
      uat: 'https://uatconnector.itrsgroup.com',
      prod: 'https://connector.itrsgroup.com',
    },
  },
};

/**
 * Determines current environment.
 *
 * @returns Current environment identifier.
 */
function getEnv() {
  let currentEnv,
    hostnameEnv = window.location.hostname.substring(0, 3);

  switch (hostnameEnv) {
    case 'loc':
      currentEnv = 'loc';
      break;

    case 'dev':
      currentEnv = 'dev';
      break;

    case 'uat':
      currentEnv = 'uat';
      break;

    default:
      currentEnv = 'prod';
      break;
  }

  return currentEnv;
}

/**
 * Get ITRS projects base URL.
 *
 * @param {string} project Project name.
 */
function getBaseUrl(project) {
  return globalConfig.baseUrl[project][getEnv()];
}

/**
 * Perfom AJAX POST request.
 *
 * @param {string} url URL to send the POST request.
 * @param {object} headers Request headers object.
 * @param {string} payload Data string to send.
 * @param {function} successCallback Callback to run on success (optional).
 * @param {function} errorCallback Callback to run on error (optional).
 */
function ajaxPost(url, headers, payload, successCallback, errorCallback) {
  let xhr = new XMLHttpRequest();

  xhr.open('POST', url);

  let headerKeys = Object.keys(headers);
  for (let i = 0; i < headerKeys.length; i++) {
    xhr.setRequestHeader(headerKeys[i], headers[headerKeys[i]]);
  }

  xhr.onreadystatechange = function() {
    // Call a function when the state changes.
    if (this.readyState === XMLHttpRequest.DONE) {
      if (this.status === 200) {
        if (typeof successCallback === 'function') {
          successCallback({
            status: this.status,
            response: this.response,
          });
        } else {
          console.log('success', this.status, this.response);
        }
      } else {
        if (typeof errorCallback === 'function') {
          errorCallback({
            status: this.status,
            response: this.response,
          });
        } else {
          console.log('something went wrong', this.status, this.response);
        }
      }
    }
  };

  xhr.send(payload);
}

/**
 * Transform params object into URL params.
 *
 * @param {object} params Params object.
 * @returns {string} URL encoded params.
 */
function urlEncodeParams(params) {
  return Object.keys(params).map(function(key) {
    if (typeof params[key] === 'object') {
      // Multivalued param.
      return Object.keys(params[key]).map(function(subKey) {
        return urlEncodeParam(key + '[]', params[key][subKey]);
      }).join('&');
    } else {
      return urlEncodeParam(key, params[key]);
    }
  }).join('&');
}

/**
 * Transform single key/value param into URL param.
 *
 * @param {string} key Param key.
 * @param {string} value Param value.
 * @returns {string} URL encoded param.
 */
function urlEncodeParam(key, value) {
  return encodeURIComponent(key) + '=' + encodeURIComponent(value);
}

/**
 * Fade out effect.
 *
 * @param {object} element Element to apply effect to.
 * @param {function} completeCallback Callback to run after effect ends.
 */
function fadeOutElement(element, completeCallback) {
  // Beware! Keep duration in sync with css animation defined for fade-out class.
  let duration = 600;
  element.classList.add('fade-out');
  setTimeout(function() { hideElement(element); }, duration);
  if (typeof completeCallback === 'function') {
    setTimeout(completeCallback, duration);
  }
}

/**
 * Show element.
 *
 * @param {object} element Element to show.
 * @param {object} removeExtraClasses Array of extra classes to remove.
 */
function showElement(element, removeExtraClasses) {
  for (let i = 0; i < removeExtraClasses.length; i++) {
    element.classList.remove(removeExtraClasses[i]);
  }

  element.classList.remove('hidden');
}

/**
 * Hide element.
 *
 * @param {object} element Element to hide.
 */
function hideElement(element) {
  element.classList.add('hidden');
}

/**
 * Disable element if condition is met.
 *
 * @param {object} element Element to set disabled to.
 * @param {boolean} disable Condition for disabled attribute.
 */
function disableElement(element, disable) {
  if (disable) {
    element.setAttribute('disabled', 'disabled');
  } else {
    element.removeAttribute('disabled');
  }
}
