-- ################################################################################
-- #                                 commands.lua                                 #
-- ################################################################################
-- # This example script shows how to publish several different types of command. #
-- # It also shows how to respond to execution and return results to the user.    #
-- ################################################################################

local gs  = require "geneos.sampler"
local cmd = require "geneos.sampler.commands"

-- Initialise a view with some fixed table data, and a headline variable 'count'.
local count = 0
local v = assert(gs.createView("myView", { "Name", "Value", "Type" }))
v.row["first row"]           = {"this is the" , "first row"     }
v.row["last row"]            = {"this is the" , "last row"      }
v.row["first row#subRow001"] = {"this is the" , "first sub row" }
v.row["first row#subRow002"] = {"this is the" , "second sub row"}
v.headline.count = count
assert(v:publish())

-- Every sample the count headline will increment by 1.
gs.doSample = function()
    count = count + 1
    print(gs.name .. " - Sample " .. count)
    v.headline.count = count
    v:publish()
end



-- A command definition that applies to the 'count' headline of view 'v'.
local h1 = cmd.newDefinition():addHeadlineTarget(v, "count")

-- Add the command 'Reset Count' using this command definition.
assert(gs.publishCommand(
    "Reset Count",               -- The "short" command name, appears in the UI menu.
    function(target, args)       -- The function to be executed is defined inline.
        count = 0                   
        v.headline.count = count -- Reset the headline count back to 0.
        assert(v:publish())
    end,
    h1                           -- Pass the command target definition.
))



-- Add a command on all rows matching the given pattern
assert(gs.publishCommand(
    "Remove Row",
    function(target, args)
        v.row[target.row] = nil -- Remove the row the user clicked on
        assert(v:publish())
    end,
    -- Command applies to all 'subrows' (i.e. rows whose name contains a # character)
    cmd.newDefinition():addRowTarget(v, "*#*")
))



-- A command with a custom menu path
-- This only affects how the command appears in the UI, the command name remains unchanged
assert(gs.publishCommand(
    "Do Nothing Column Command",
    function(target, args) --[[ Does nothing ]] end,
    cmd.newDefinition()
        :setMenuPath( {"Nested","Menu","Command"} )
        :addColumnTarget(v, "Value")
))



-- A multi-argument command, defined using the fluent interface
local argsFluent = cmd.newDefinition()
    :setDescription("An example of various command argument types.")   -- Description appears at the top of user input dialog
    :addStringArgument("myString", "My String Default")                -- String  argument, with default
    :addIntegerArgument("myInteger", 123456)                           -- Integer argument, with default
    :addNumericArgument("myDouble", 123.456)                           -- Double  arguemnt, with default
    :addBooleanArgument("myBoolean", true)                             -- Boolean argument, with default
    :addChoiceArgument("myChoice", {"Red", "Yellow", "Green", "Blue"}) -- Choice  argument, with 4 options specified (first is default)
    :addRowTarget(v, '*')                                              -- Applies to all rows

-- The same command definition, in a tabular style
local argsTabular = {
    description = "An example of various command argument types.",
    args = {
        { type="string",  description="myString",  default="My String Default" },
        { type="integer", description="myInteger", default=123456 },
        { type="number",  description="myDouble",  default=123.456 },
        { type="boolean", description="myBoolean", default=true },
        { type="choice",  description="myChoice",  options={"Red", "Yellow", "Green", "Blue"} }
    },
    targets = {
        { type="row", view="myView", name="*" }
    }
}

-- Shows the arguments of the callback function in the view when executed
assert(gs.publishCommand(
    "Show Callback Args",
    function(target, args)
        v.row = {}
        -- Add all target parameters to the view, one row per parameter
        for key,value in pairs(target) do
            v.row[string.format("target.%s", key)] = { tostring(value), type(value) }
        end
        -- All all arguments to the view, one row per argument value
        for i,value in ipairs(args) do
            v.row[string.format("args[%d]", i)] = { tostring(value), type(value) }
        end
        assert(v:publish())
    end,
    argsTabular
))



-- A command which outputs a message
-- Since no targets are specified, this command is only accessible from the dataview context menu
assert(gs.publishCommand(
    "Output Message",
    function(target, args) return "This is a\nmulti\nline\nmessage" end
))



-- Base command definition
local baseDef = cmd.newDefinition():addStringArgument("Name")

assert(gs.publishCommand(
    "Say Hello",
    function(target, args) return "Hello " .. args[1] end,
    -- Derived definition, with altered description
    baseDef:new():setDescription("Say hello to ...")
))

assert(gs.publishCommand(
    "Say Goodbye",
    function(target, args) return "Goodbye " .. args[1] end,
    -- Derived definition, with altered description
    baseDef:new():setDescription("Say goodbye to ...")
))
