document
    .addEventListener('DOMContentLoaded', inPageSearch, false); // Initialize the inPageSearch function after the HTML is fully loaded.

function ipsAddEventListener(el, eventName, eventHandler, selector) {
    // A utility function to add event listeners with optional event delegation.
    let wrappedHandler;


    if (selector) {
        wrappedHandler = (e) => {
            if (!e.target) return;
            const el = e.target.closest(selector); // Find the closest ancestor matching the selector.
            if (el) {
                eventHandler.call(el, e); // Call the event handler in the context of the matched element.
            }
        };
    } else {
        wrappedHandler = (e) => {
            eventHandler.call(el, e); // Call the event handler directly on the element.
        };
    }

    el.addEventListener(eventName, wrappedHandler); // Attach the event listener.

    return wrappedHandler; // Return the wrapped handler for potential removal or reuse.
}

function pageSearchBtnFilterClick() {


    // Toggles the 'active' state of filter buttons and triggers the filter function.
    var $button = this;

    $button.classList.toggle('active'); // Add or remove 'active' class.

    pageSearchFilter(); // Apply filters.
}

function pageSearchFilterClearAllClick() {
    // Clears all active filters and resets the search input field.
    document.querySelectorAll('.btn-filter.active').forEach((el, i) => {
        el.classList.remove('active'); // Remove 'active' class from all filter buttons.
    });
    document.querySelector('.plugin-search-filter').value = ''; // Clear the search input field.
    pageSearchFilter(); // Refresh the filtering logic.
}

function searchInFilters(filters, $filterDiv) {
    // Checks if a filterDiv matches any of the active filters.

    return filters.filter(function (f) {
        return $filterDiv.dataset.filter.split(',').indexOf(f) >= 0; // Match filters against data-filter attribute.
    }).length; // Return the count of matches.
}

const fusePluginsOption = {
    // Configuration object for Fuse.js fuzzy search.
    includeScore: true, // Include relevance scores in search results.
    shouldSort: true, // Sort results by relevance.
    keys: [{
        name: 'data_title',
        weight: 100 // High relevance weight for title.
    },
    {
        name: 'data_keyword',
        weight: 100 // High relevance weight for keywords.
    },
    ],
    threshold: 0.3, // Strict matching threshold for higher precision.
    distance: 700
};

function filterDivData($filterDiv) {
    // Extracts relevant data from a filterDiv element into a format usable by Fuse.js.
    return [{
        data_title: $filterDiv.querySelector('.filterDiv-title').textContent, // Title from the tooltip dataset.
        data_keyword: $filterDiv.dataset.keywords // Keywords from the dataset.
    }];
}

function searchInText($filterDiv, searchText) {
    // Performs a fuzzy search on a filterDiv using the provided search text.
    const pluginsFuse = new Fuse(filterDivData($filterDiv), fusePluginsOption); // Initialize Fuse.js.

    const results = pluginsFuse.search(searchText); // Search for matches.

    return results.length > 0; // Return true if matches are found, otherwise false.
}

const queryString = window.location.search; // access page url location search

const urlParams = new URLSearchParams(queryString); // parse search params

function checkParamsURL() {

    const getParamsFilterValues = urlParams.getAll("pre-filter"); // get the value of the params
    const getParamsSearchValue = urlParams.get("pre-search"); // get the value of the params

    if (getParamsFilterValues.length > 0 || getParamsSearchValue) {

        applyParamsFilter(getParamsFilterValues, getParamsSearchValue); // pass value of params to search filter function
    }
}

function applyParamsFilter(paramsFilterValue, paramsSearchValue) { // apply filter from params
    if (paramsFilterValue.length > 0) {
        paramsFilterValue.forEach((params) => {
            document.querySelectorAll('.btn-filter').forEach((el) => {
                if (el.dataset.filter === params) {
                    el.classList.add('active')
                }
            })
        })
    }

    if (paramsSearchValue) {
        document.querySelector(".plugin-search-filter").value = decodeURIComponent(paramsSearchValue) // decode pre search params
    }

    pageSearchFilter()
}

function pageSearchFilter() {
    // Applies filtering logic to display only elements matching the active filters or search text.
    var filters = [];
    var searchText = document.querySelector('.plugin-search-filter') ?
        document.querySelector('.plugin-search-filter').value.toLowerCase() : // Get the search text input (case-insensitive).
        '';
    var resultsCount = 0;



    document.querySelectorAll('.btn-filter.active').forEach((el, i) => {
        filters.push(el.dataset.filter); // Collect all active filters.
    });


    if (filters.length > 0 || searchText) { // append params to browser url when selecting filters
        urlParams.delete('pre-filter')
        filters.forEach((e) => {
            urlParams.append('pre-filter', e)
        })

        window.history.replaceState({}, '', '?' + urlParams.toString());
        urlParams.set('pre-search', encodeURIComponent(searchText)) // encode search text for url params
    } else {
        window.history.pushState({}, '', window.location.pathname);
    }

    document.querySelectorAll('.filterDiv').forEach((el, i) => {
        // Iterate through all filterDiv elements and determine visibility.
        var $filterDiv = el;

        if ((filters.length == 0 || searchInFilters(filters, $filterDiv)) &&
            (!searchText || searchInText($filterDiv, searchText))) {
            $filterDiv.classList.remove('element-invisible'); // Show matching elements.
            resultsCount++;

        } else {
            $filterDiv.classList.add('element-invisible'); // Hide non-matching elements.
        }
    });

    document // Toggle visibility of the "No results found." message.
        .querySelector('.no-results')
        .classList
        .toggle('element-invisible', resultsCount !== 0);
}


function inPageSearch() {

    checkParamsURL() // check URL params for pre filtering

    // Initializes event listeners for filtering and searching functionality.
    ipsAddEventListener(
        document,
        'click',
        pageSearchBtnFilterClick,
        '.btn-filter' // Attach to filter buttons.
    );

    ipsAddEventListener(
        document,
        'click',
        pageSearchFilterClearAllClick,
        '.filter-clear-all' // Attach to the "Clear All" button.
    );

    ipsAddEventListener(
        document,
        'keyup',
        pageSearchFilter,
        '.plugin-search-filter' // Attach to the search input field.
    );
}